<?php
/**
 * @author     Bart de Groot
 * @company    Abovo Media
 * @copyright  Copyright (c) 2017 Abovo Media (http://www.abovomedia.nl)
 * @package    Totem_Widgets
 */

namespace Totem\Widgets\Block\Widget;

use Magento\Framework\App\ObjectManager;
use Magento\Framework\Serialize\Serializer\Json;

class Products extends \Magento\CatalogWidget\Block\Product\ProductsList
{
   const DEFAULT_SORT_BY         = 'entity_id';
   const DEFAULT_SORT_ORDER      = 'asc';
   const DEFAULT_PAGE_SIZE       = 12;
   
   const RANDOM_SORT_BY          = 'random';
   
   /**
    * @var \Magento\Framework\Url\Helper\Data
    */
   protected $urlHelper;
   
   /**
    * Json Serializer Instance
    *
    * @var Json
    */
   protected $json;
   
   /**
    * ProductList constructor.
    * @param \Magento\Catalog\Block\Product\Context $context
    * @param \Magento\Catalog\Model\ResourceModel\Product\CollectionFactory $productCollectionFactory
    * @param \Magento\Catalog\Model\Product\Visibility $catalogProductVisibility
    * @param \Magento\Framework\App\Http\Context $httpContext
    * @param \Magento\Rule\Model\Condition\Sql\Builder $sqlBuilder
    * @param \Magento\CatalogWidget\Model\Rule $rule
    * @param \Magento\Widget\Helper\Conditions $conditionsHelper
    * @param \Magento\Catalog\Api\CategoryRepositoryInterface $categoryRepository
    * @param \Magento\Framework\Url\Helper\Data $urlHelper
    * @param array $data
    * @param Json|null $json
    * @param LayoutFactory|null $layoutFactory
    * @param \Magento\Framework\Url\EncoderInterface|null $urlEncoder
    */
   public function __construct(
      \Magento\Catalog\Block\Product\Context $context,
      \Magento\Catalog\Model\ResourceModel\Product\CollectionFactory $productCollectionFactory,
      \Magento\Catalog\Model\Product\Visibility $catalogProductVisibility,
      \Magento\Framework\App\Http\Context $httpContext,
      \Magento\Rule\Model\Condition\Sql\Builder $sqlBuilder,
      \Magento\CatalogWidget\Model\Rule $rule,
      \Magento\Widget\Helper\Conditions $conditionsHelper,
      array $data = [],
      Json $json = null,
      \Magento\Framework\View\LayoutFactory $layoutFactory = null,
      \Magento\Framework\Url\EncoderInterface $urlEncoder = null,
      \Magento\Catalog\Api\CategoryRepositoryInterface $categoryRepository,
      \Magento\Framework\Url\Helper\Data $urlHelper
   ) {
      $this->urlHelper  = $urlHelper;
      $this->json       = $json ?: ObjectManager::getInstance()->get(Json::class);
   
      parent::__construct($context, $productCollectionFactory, $catalogProductVisibility, $httpContext, $sqlBuilder, $rule, $conditionsHelper, $data, $json, $layoutFactory, $urlEncoder, $categoryRepository);
   }
   
   public function createCollection()
   {
      $collection = $this->productCollectionFactory->create();
      
      $collection = $this->_addProductAttributesAndPrices($collection)
                         ->addStoreFilter()
                         ->setPageSize($this->_getPageSize());
   
      $collection->setVisibility($this->catalogProductVisibility->getVisibleInCatalogIds());
      
      if($this->_getSortBy() == static::RANDOM_SORT_BY):
         $collection->getSelect()->orderRand();
      else:
         $collection->setOrder($this->_getSortBy(), $this->_getSortOrder());
      endif;
      
      $conditions = $this->getConditions();
      $conditions->collectValidatedAttributes($collection);
      $this->sqlBuilder->attachConditionToCollection($collection, $conditions);
      
      return $collection;
   }
   
   /**
    * @param null $type
    *
    * @return bool|\Magento\Framework\View\Element\AbstractBlock|null
    */
   public function getDetailsRenderer($type = null)
   {
      if($type === null):
         $type = 'default';
      endif;
      
      $rendererList = $this->getDetailsRendererList();
      if ($rendererList):
         return $rendererList->getRenderer($type, 'default');
      endif;
      
      return null;
   }
   
   /**
    * @return bool|\Magento\Framework\View\Element\BlockInterface
    */
   protected function getDetailsRendererList()
   {
      return $this->getDetailsRendererListName()
         ? $this->getLayout()->getBlock($this->getDetailsRendererListName())
         : $this->getLayout()->createBlock('Totem\Widgets\Block\View\Element\CustomRendererList')->setWidgetName('totem_widget_products');
   }
   
   /**
    * @param \Magento\Catalog\Model\Product $product
    *
    * @return array
    */
   public function getAddToCartPostParams(\Magento\Catalog\Model\Product $product)
   {
      $url = $this->getAddToCartUrl($product);
      return [
         'action' => $url,
         'data' => [
            'product'                                                      => $product->getEntityId(),
            \Magento\Framework\App\ActionInterface::PARAM_NAME_URL_ENCODED => $this->urlHelper->getEncodedUrl($url)
         ]
      ];
   }
   
   /**
    * Retrieve how many products should be displayed
    *
    * @return int
    */
   private function _getPageSize()
   {
      if (!$this->hasData('products_page_size')):
         $this->setData('products_page_size', static::DEFAULT_PAGE_SIZE);
      endif;
      
      return $this->getData('products_page_size');
   }
   
   /**
    * Retrieve how the products should be sorted
    *
    * @return string
    */
   private function _getSortBy()
   {
      if (!$this->hasData('products_sort_by')):
         $this->setData('products_sort_by', static::DEFAULT_SORT_BY);
      endif;
      
      return $this->getData('products_sort_by');
   }
   
   /**
    * Retrieve how the products should be ordered
    *
    * @return string
    */
   private function _getSortOrder()
   {
      if (!$this->hasData('products_sort_order')):
         $this->setData('products_sort_order', static::DEFAULT_SORT_ORDER);
      endif;
      
      return $this->getData('products_sort_order');
   }
}