<?php
/**
 * @author     Bart de Groot
 * @company    Abovo Media
 * @copyright  Copyright (c) 2017 Abovo Media (http://www.abovomedia.nl)
 * @package    Totem_Settings
 */

namespace Totem\Settings\Setup;

use Magento\Framework\Setup\InstallDataInterface;
use Magento\Framework\Setup\ModuleContextInterface;
use Magento\Framework\Setup\ModuleDataSetupInterface;

/**
 * @codeCoverageIgnore
 */
class InstallData implements InstallDataInterface
{
   /**
    * {@inheritdoc}
    *
    * @SuppressWarnings(PHPMD.ExcessiveMethodLength)
    */
   public function install(
      ModuleDataSetupInterface $setup,
      ModuleContextInterface $context
   ) {
      $setup->startSetup();
      
      $connection = $setup->getConnection();
      
      if(!$this->_checkPreconditions($connection)):
         $setup->endSetup();
         
         return;
      endif;
      
      $taxClassData = $this->_getTaxClassData();
      try
      {
         foreach($taxClassData as $taxClass):
            $connection->insertOnDuplicate($setup->getTable('tax_class'), $taxClass, ['class_name', 'class_type']);
         endforeach;
      }
      catch(\Exception $e)
      {
         $connection->rollBack();
      }
      
      $taxCalculationRuleData = $this->_getTaxCalculationRuleData();
      try
      {
         foreach($taxCalculationRuleData as $taxCalculationRule):
            $connection->insertOnDuplicate($setup->getTable('tax_calculation_rule'), $taxCalculationRule, ['code']);
         endforeach;
      }
      catch(\Exception $e)
      {
         $connection->rollBack();
      }
      
      $taxCalculationRateData = $this->_getTaxCalculationRateData();
      try
      {
         $connection->truncateTable('tax_calculation_rate');
         
         $connection->beginTransaction();
         $connection->insertMultiple($setup->getTable('tax_calculation_rate'), $taxCalculationRateData);
         $connection->commit();
      }
      catch(\Exception $e)
      {
         $connection->rollBack();
      }
      
      $taxCalculationData = $this->_getTaxCalculationData();
      try
      {
         $connection->truncateTable('tax_calculation');
         
         $connection->beginTransaction();
         $connection->insertMultiple($setup->getTable('tax_calculation'), $taxCalculationData);
         $connection->commit();
      }
      catch(\Exception $e)
      {
         $connection->rollBack();
      }
      
      $setup->endSetup();
   }
   
   /**
    * Preconditions to continue
    *
    * @param \Magento\Framework\DB\Adapter\AdapterInterface $connection
    *
    * @return bool
    */
   private function _checkPreconditions(\Magento\Framework\DB\Adapter\AdapterInterface $connection)
   {
      $tables = [
         'tax_class',
         'tax_calculation_rule',
         'tax_calculation_rate',
         'tax_calculation',
      ];
      
      foreach($tables as $table):
         if(!$connection->isTableExists($table)):
            return false;
         endif;
         
         continue;
      endforeach;
      
      $query = 'SELECT `tax_country_id` FROM `tax_calculation_rate` WHERE `tax_calculation_rate_id` = 1 LIMIT 1';
      
      if($connection->fetchOne($query) == 'NL'):
         return false;
      endif;
      
      return true;
   }
   
   /**
    * Get the data for the tax_class table
    *
    * @return array
    */
   private function _getTaxClassData()
   {
      $data = [];
      $data[] = ['class_id' => '2', 'class_name' => 'Standard',        'class_type' => 'PRODUCT'];
      $data[] = ['class_id' => '3', 'class_name' => 'Retail Customer', 'class_type' => 'CUSTOMER'];
      $data[] = ['class_id' => '4', 'class_name' => 'Reduced',         'class_type' => 'PRODUCT'];
      $data[] = ['class_id' => '5', 'class_name' => 'Zero',            'class_type' => 'PRODUCT'];
      
      return $data;
   }
   
   /**
    * Get the data for the tax_calculation_rule table
    *
    * @return array
    */
   private function _getTaxCalculationRuleData()
   {
      $data = [];
      $data[] = ['tax_calculation_rule_id' => '2', 'code' => 'Standard', 'priority' => 0, 'position' => 0, 'calculate_subtotal' => 0];
      $data[] = ['tax_calculation_rule_id' => '3', 'code' => 'Reduced',  'priority' => 0, 'position' => 0, 'calculate_subtotal' => 0];
      $data[] = ['tax_calculation_rule_id' => '4', 'code' => 'Zero',     'priority' => 0, 'position' => 0, 'calculate_subtotal' => 0];
      
      return $data;
   }
   
   /**
    * Get the data for the tax_calculation_rate table
    *
    * @return array
    */
   private function _getTaxCalculationRateData()
   {
      $data = [];
      $data[] = ['tax_calculation_rate_id' => '1',  'tax_country_id' => 'AT', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Austria (standard)',        'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '2',  'tax_country_id' => 'AT', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Austria (reduced)',         'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '3',  'tax_country_id' => 'AT', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Austria (zero)',            'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '4',  'tax_country_id' => 'BE', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Belgium (standard)',        'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '5',  'tax_country_id' => 'BE', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Belgium (reduced)',         'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '6',  'tax_country_id' => 'BE', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Belgium (zero)',            'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '7',  'tax_country_id' => 'BG', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Bulgaria (standard)',       'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '8',  'tax_country_id' => 'BG', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Bulgaria (reduced)',        'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '9',  'tax_country_id' => 'BG', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Bulgaria (zero)',           'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '10', 'tax_country_id' => 'HR', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Croatia (standard)',        'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '11', 'tax_country_id' => 'HR', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Croatia (reduced)',         'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '12', 'tax_country_id' => 'HR', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Croatia (zero)',            'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '13', 'tax_country_id' => 'CY', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Cyprus (standard)',         'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '14', 'tax_country_id' => 'CY', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Cyprus (reduced)',          'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '15', 'tax_country_id' => 'CY', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Cyprus (zero)',             'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '16', 'tax_country_id' => 'CZ', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Czech Republic (standard)', 'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '17', 'tax_country_id' => 'CZ', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Czech Republic (reduced)',  'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '18', 'tax_country_id' => 'CZ', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Czech Republic (zero)',     'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '19', 'tax_country_id' => 'DK', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Denmark (standard)',        'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '20', 'tax_country_id' => 'DK', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Denmark (reduced)',         'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '21', 'tax_country_id' => 'DK', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Denmark (zero)',            'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '22', 'tax_country_id' => 'EE', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Estonia (standard)',        'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '23', 'tax_country_id' => 'EE', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Estonia (reduced)',         'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '24', 'tax_country_id' => 'EE', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Estonia (zero)',            'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '25', 'tax_country_id' => 'FI', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Finland (standard)',        'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '26', 'tax_country_id' => 'FI', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Finland (reduced)',         'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '27', 'tax_country_id' => 'FI', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Finland (zero)',            'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '28', 'tax_country_id' => 'FR', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'France (standard)',         'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '29', 'tax_country_id' => 'FR', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'France (reduced)',          'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '30', 'tax_country_id' => 'FR', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'France (zero)',             'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '31', 'tax_country_id' => 'DE', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Germany (standard)',        'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '32', 'tax_country_id' => 'DE', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Germany (reduced)',         'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '33', 'tax_country_id' => 'DE', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Germany (zero)',            'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '34', 'tax_country_id' => 'GR', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Greece (standard)',         'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '35', 'tax_country_id' => 'GR', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Greece (reduced)',          'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '36', 'tax_country_id' => 'GR', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Greece (zero)',             'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '37', 'tax_country_id' => 'HU', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Hungary (standard)',        'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '38', 'tax_country_id' => 'HU', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Hungary (reduced)',         'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '39', 'tax_country_id' => 'HU', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Hungary (zero)',            'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '40', 'tax_country_id' => 'IE', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Ireland (standard)',        'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '41', 'tax_country_id' => 'IE', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Ireland (reduced)',         'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '42', 'tax_country_id' => 'IE', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Ireland (zero)',            'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '43', 'tax_country_id' => 'IT', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Italy (standard)',          'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '44', 'tax_country_id' => 'IT', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Italy (reduced)',           'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '45', 'tax_country_id' => 'IT', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Italy (zero)',              'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '46', 'tax_country_id' => 'LV', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Latvia (standard)',         'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '47', 'tax_country_id' => 'LV', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Latvia (reduced)',          'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '48', 'tax_country_id' => 'LV', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Latvia (zero)',             'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '49', 'tax_country_id' => 'LT', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Lithuania (standard)',      'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '50', 'tax_country_id' => 'LT', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Lithuania (reduced)',       'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '51', 'tax_country_id' => 'LT', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Lithuania (zero)',          'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '52', 'tax_country_id' => 'LU', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Luxembourg (standard)',     'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '53', 'tax_country_id' => 'LU', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Luxembourg (reduced)',      'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '54', 'tax_country_id' => 'LU', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Luxembourg (zero)',         'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '55', 'tax_country_id' => 'MT', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Malta (standard)',          'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '56', 'tax_country_id' => 'MT', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Malta (reduced)',           'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '57', 'tax_country_id' => 'MT', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Malta (zero)',              'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '58', 'tax_country_id' => 'NL', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Netherlands (standard)',    'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '59', 'tax_country_id' => 'NL', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Netherlands (reduced)',     'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '60', 'tax_country_id' => 'NL', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Netherlands (zero)',        'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '61', 'tax_country_id' => 'PL', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Poland (standard)',         'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '62', 'tax_country_id' => 'PL', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Poland (reduced)',          'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '63', 'tax_country_id' => 'PL', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Poland (zero)',             'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '64', 'tax_country_id' => 'PT', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Portugal (standard)',       'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '65', 'tax_country_id' => 'PT', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Portugal (reduced)',        'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '66', 'tax_country_id' => 'PT', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Portugal (zero)',           'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '67', 'tax_country_id' => 'RO', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Romania (standard)',        'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '68', 'tax_country_id' => 'RO', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Romania (reduced)',         'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '69', 'tax_country_id' => 'RO', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Romania (zero)',            'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '70', 'tax_country_id' => 'SK', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Slovakia (standard)',       'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '71', 'tax_country_id' => 'SK', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Slovakia (reduced)',        'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '72', 'tax_country_id' => 'SK', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Slovakia (zero)',           'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '73', 'tax_country_id' => 'SK', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Slovenia (standard)',       'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '74', 'tax_country_id' => 'SK', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Slovenia (reduced)',        'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '75', 'tax_country_id' => 'SK', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Slovenia (zero)',           'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '76', 'tax_country_id' => 'ES', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Spain (standard)',          'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '77', 'tax_country_id' => 'ES', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Spain (reduced)',           'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '78', 'tax_country_id' => 'ES', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Spain (zero)',              'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '79', 'tax_country_id' => 'SE', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Sweden (standard)',         'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '80', 'tax_country_id' => 'SE', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Sweden (reduced)',          'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '81', 'tax_country_id' => 'SE', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'Sweden (zero)',             'rate' => 0.0000];
      $data[] = ['tax_calculation_rate_id' => '82', 'tax_country_id' => 'GB', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'United Kingdom (standard)', 'rate' => 21.0000];
      $data[] = ['tax_calculation_rate_id' => '83', 'tax_country_id' => 'GB', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'United Kingdom (reduced)',  'rate' => 9.0000];
      $data[] = ['tax_calculation_rate_id' => '84', 'tax_country_id' => 'GB', 'tax_region_id' => 0, 'tax_postcode' => '*', 'code' => 'United Kingdom (zero)',     'rate' => 0.0000];
      
      return $data;
   }
   
   /**
    * Get the data for the tax_calculation table
    *
    * @return array
    */
   private function _getTaxCalculationData()
   {
      $data = [];
      $data[] = ['tax_calculation_rate_id' => 1,  'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 4,  'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 7,  'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 10, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 13, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 16, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 19, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 22, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 25, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 28, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 31, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 34, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 37, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 40, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 43, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 46, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 49, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 52, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 55, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 58, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 61, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 64, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 67, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 70, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 73, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 76, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 79, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 82, 'tax_calculation_rule_id' => 2, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 2];
      $data[] = ['tax_calculation_rate_id' => 2,  'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 5,  'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 8,  'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 11, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 14, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 17, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 20, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 23, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 26, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 29, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 32, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 35, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 38, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 41, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 44, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 47, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 50, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 53, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 56, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 59, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 62, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 65, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 68, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 71, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 74, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 77, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 80, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 83, 'tax_calculation_rule_id' => 3, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 4];
      $data[] = ['tax_calculation_rate_id' => 3,  'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 6,  'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 9,  'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 12, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 15, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 18, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 21, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 24, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 27, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 30, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 33, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 36, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 39, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 42, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 45, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 48, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 51, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 54, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 57, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 60, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 63, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 66, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 69, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 72, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 75, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 78, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 81, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      $data[] = ['tax_calculation_rate_id' => 84, 'tax_calculation_rule_id' => 4, 'customer_tax_class_id' => 3, 'product_tax_class_id' => 5];
      
      return $data;
   }
}