<?php
/**
 * @author     Bart de Groot
 * @company    Abovo Media
 * @copyright  Copyright (c) 2017 Abovo Media (http://www.abovomedia.nl)
 * @package    Totem_Settings
 */

namespace Totem\Settings\Helper;

use Magento\Store\Model\ScopeInterface;
use Magento\Framework\Stdlib\DateTime\TimezoneInterface;
use Magento\Framework\App\Http\Context as HttpContext;

class Data extends \Magento\Framework\App\Helper\AbstractHelper
{
   /**
    * @var ScopeConfigInterface
    */
   protected $scopeConfig;
   
   /**
    * @var TimezoneInterface
    */
   protected $timezoneInterface;
   
   /**
    * @var HttpContext
    */
   protected $httpContext;
   
   /**
    * @param \Magento\Framework\App\Helper\Context $context
    * @param TimezoneInterface $timezoneInterface
    * @param HttpContext $httpContext
    */
   public function __construct(
      \Magento\Framework\App\Helper\Context $context,
      TimezoneInterface $timezoneInterface,
      HttpContext $httpContext
   ) {
      $this->scopeConfig         = $context->getScopeConfig();
      $this->timezoneInterface   = $timezoneInterface;
      $this->httpContext         = $httpContext;
      
      
      parent::__construct($context);
   }
   
   /**
    * Get config value
    *
    * @return string
    */
   public function getConfig($configPath, $isSetFlag = false)
   {
      if($isSetFlag):
         return $this->scopeConfig->isSetFlag($configPath, ScopeInterface::SCOPE_STORE);
      endif;
      
      return $this->scopeConfig->getValue($configPath, ScopeInterface::SCOPE_STORE);
   }
   
   /**
    * Check if variable is set
    *
    * @return boolean
    */
   public function isVariableSet($variable, $type = 'string', $trim = true)
   {
      if($trim):
         $variable = trim($variable);
      endif;
      
      switch($type):
         case 'string':
         default:
            if (!is_string($variable) || empty($variable)):
               return false;
            endif;
            
            return true;
      endswitch;
      
      return false;
   }
   
   /**
    * Get date in short format without the year
    *
    * @param string $dateTime
    * @return string $dateFormatted
    */
   public function getDateShortWithoutYear($dateTime)
   {
      $dateFormatted = $this->timezoneInterface->formatDate($dateTime, \IntlDateFormatter::SHORT);
      $dateYear      = $this->timezoneInterface->date(new \DateTime($dateTime))->format('y');
      
      return preg_replace(
         '/[-\/.]' . $dateYear . '$/',
         '',
         $dateFormatted
      );
   }
   
   /**
    * Get date in medium format without the year
    *
    * @param string $dateTime
    * @return string $dateFormatted
    */
   public function getDateMediumWithoutYear($dateTime)
   {
      $dateFormatted = $this->timezoneInterface->formatDate($dateTime, \IntlDateFormatter::MEDIUM);
      $dateYear      = $this->timezoneInterface->date(new \DateTime($dateTime))->format('Y');
      
      return preg_replace(
         '/ ' . $dateYear . '$/',
         '',
         $dateFormatted
      );
   }
   
   /**
    * Get date in long format without the year
    *
    * @param string $dateTime
    * @return string $dateFormatted
    */
   public function getDateLongWithoutYear($dateTime)
   {
      $dateFormatted = $this->timezoneInterface->formatDate($dateTime, \IntlDateFormatter::LONG);
      $dateYear      = $this->timezoneInterface->date(new \DateTime($dateTime))->format('Y');
      
      return preg_replace(
         '/ ' . $dateYear . '$/',
         '',
         $dateFormatted
      );
   }
   
   /**
    * Get date in full format without the year
    *
    * @param string $dateTime
    * @return string $dateFormatted
    */
   public function getDateFullWithoutYear($dateTime)
   {
      $dateFormatted = $this->timezoneInterface->formatDate($dateTime, \IntlDateFormatter::FULL);
      $dateYear      = $this->timezoneInterface->date(new \DateTime($dateTime))->format('Y');
      
      return preg_replace(
         '/ ' . $dateYear . '$/',
         '',
         $dateFormatted
      );
   }
   
   /**
    * Get year only from date
    *
    * @param string $dateTime
    * @return string $dateYear
    */
   public function getDateYearOnly($dateTime)
   {
      $dateYear      = $this->timezoneInterface->date(new \DateTime($dateTime))->format('Y');
      
      return $dateYear;
   }
   
   /**
    * Check if customer logged in
    *
    * @return bool
    */
   public function isLoggedIn()
   {
      return (bool)$this->httpContext->getValue(\Magento\Customer\Model\Context::CONTEXT_AUTH);
   }
}