<?php
/**
 * @author     Bart de Groot
 * @company    Abovo Media
 * @copyright  Copyright (c) 2019 Abovo Media (http://www.abovomedia.nl)
 * @package    Totem_MenuManager
 */

namespace Totem\MenuManager\Setup;

use Magento\Framework\DB\Ddl\Table;
use Magento\Framework\Setup\ModuleContextInterface;
use Magento\Framework\Setup\SchemaSetupInterface;
use Magento\Framework\Setup\UpgradeSchemaInterface;

class UpgradeSchema implements UpgradeSchemaInterface
{
   /**
    * {@inheritdoc}
    */
   public function upgrade(SchemaSetupInterface $setup, ModuleContextInterface $context)
   {
      $setup->startSetup();

      if (version_compare($context->getVersion(), '2.0.0', '<')):
         $this->_renameTables($setup);
         
         $this->_changeColumns($setup);
         
         $this->_addColumns($setup);
         
         $this->_dropColumns($setup);
         
         $this->_deprecateColumns($setup);
         
         $this->_updateTargetAttribute($setup);
      endif;
      
      $setup->endSetup();
   }
   
   private function _renameTables(SchemaSetupInterface $setup)
   {
      $connection = $setup->getConnection();
      
      $oldTable   = $setup->getTable('totem_menumanager_item');
      $newTable   = $setup->getTable('totem_menumanager_node');
   
      if ($setup->tableExists('totem_menumanager_item')) {
         $connection->renameTable($oldTable, $newTable);
      }
      
      $oldTable   = $setup->getTable('totem_menumanager_menu_store');
      $newTable   = $setup->getTable('totem_menumanager_store');
   
      if ($setup->tableExists('totem_menumanager_menu_store')) {
         $connection->renameTable($oldTable, $newTable);
      }
   }
   
   private function _updateTargetAttribute(SchemaSetupInterface $setup)
   {
      $table      = $setup->getTable('totem_menumanager_node');
      $connection = $setup->getConnection();
      
      $connection->modifyColumn(
         $table,
         'target',
         [
            'type'      => Table::TYPE_BOOLEAN,
            'default'   => 0,
            'comment'   => 'Link Target'
         ]
      );
   }
   
   private function _changeColumns(SchemaSetupInterface $setup)
   {
      $table      = $setup->getTable('totem_menumanager_node');
      $connection = $setup->getConnection();
   
      $connection->changeColumn(
         $table,
         'item_id',
         'node_id',
         [
            'type'      => Table::TYPE_INTEGER,
            'identity'  => true,
            'nullable'  => false,
            'primary'   => true,
            'unsigned'  => true,
            'comment'   => 'Node ID'
         ]
      );
   
      $connection->changeColumn(
         $table,
         'url_type',
         'type',
         [
            'type'      => Table::TYPE_TEXT,
            'length'    => 255,
            'nullable'  => false,
            'default'   => 0,
            'comment'   => 'Node Type'
         ]
      );
      
      $connection->changeColumn(
         $table,
         'url',
         'content',
         [
            'type'      => Table::TYPE_TEXT,
            'after'     => 'type',
            'comment'   => 'Node contents'
         ]
      );
   
      $connection->update(
         $table,
         ['type' => 'custom_url'],
         "type = '0'"
      );
      $connection->update(
         $table,
         ['type' => 'cms_page'],
         "type = '1'"
      );
      $connection->update(
         $table,
         ['type' => 'category'],
         "type = '2'"
      );
   
      $connection->changeColumn(
         $table,
         'auto_subcategories',
         'subcategories',
         [
            'type'      => Table::TYPE_BOOLEAN,
            'default'   => 0
         ]
      );
      
      return $this;
   }
   
   private function _addColumns(SchemaSetupInterface $setup)
   {
      $tableNode = $setup->getTable('totem_menumanager_node');
      $tableMenu = $setup->getTable('totem_menumanager_menu');
   
      /**
       * TABLE: totem_menumanager_node
       */
      $setup->getConnection()->addColumn(
         $tableNode,
         'classes',
         [
            'type'      => Table::TYPE_TEXT,
            'length'    => 255,
            'after'     => 'subcategories',
            'comment'   => 'CSS class name'
         ]
      );
      
      $setup->getConnection()->addColumn(
         $tableNode,
         'level',
         [
            'type'      => Table::TYPE_INTEGER,
            'length'    => null,
            'nullable'  => false,
            'unsigned'  => true,
            'after'     => 'position',
            'comment'   => 'Node level'
         ]
      );
      
      $setup->getConnection()->addColumn(
         $tableNode,
         'creation_time',
         [
            'type'      => Table::TYPE_TIMESTAMP,
            'length'    => null,
            'nullable'  => false,
            'after'     => 'level',
            'default'   => Table::TIMESTAMP_INIT,
            'comment'   => 'Creation Time'
         ]
      );
      
      $setup->getConnection()->addColumn(
         $tableNode,
         'update_time',
         [
            'type'      => Table::TYPE_TIMESTAMP,
            'length'    => null,
            'nullable'  => false,
            'after'     => 'creation_time',
            'default'   => Table::TIMESTAMP_INIT_UPDATE,
            'comment'   => 'Modification Time'
         ]
      );
   
      /**
       * TABLE: totem_menumanager_menu
       */
      $setup->getConnection()->addColumn(
         $tableMenu,
         'creation_time',
         [
            'type'      => Table::TYPE_TIMESTAMP,
            'length'    => null,
            'nullable'  => false,
            'after'     => 'css_class',
            'default'   => Table::TIMESTAMP_INIT,
            'comment'   => 'Creation Time'
         ]
      );
   
      $setup->getConnection()->addColumn(
         $tableMenu,
         'update_time',
         [
            'type'      => Table::TYPE_TIMESTAMP,
            'length'    => null,
            'nullable'  => false,
            'after'     => 'creation_time',
            'default'   => Table::TIMESTAMP_INIT_UPDATE,
            'comment'   => 'Modification Time'
         ]
      );
      
      return $this;
   }
   
   /**
    * @param SchemaSetupInterface $setup
    * @return $this
    */
   private function _dropColumns(SchemaSetupInterface $setup)
   {
      $tableNode = $setup->getTable('totem_menumanager_node');
      
      $setup->getConnection()->dropColumn(
         $tableNode,
         'identifier'
      );
      
      return $this;
   }
   
   /**
    * @param SchemaSetupInterface $setup
    * @return $this
    */
   private function _deprecateColumns(SchemaSetupInterface $setup)
   {
      $tableNode = $setup->getTable('totem_menumanager_node');
   
      /**
       * Drop deprecated foreign keys
       */
      $foreignKeys = $setup->getConnection()->getForeignKeys('totem_menumanager_node');
      foreach($foreignKeys as $fkData):
         if(!in_array($fkData['COLUMN_NAME'], ['cms_page_identifier', 'category_id'])):
            continue;
         endif;
   
         $setup->getConnection()->dropForeignKey(
            $tableNode,
            $fkData['FK_NAME']
         );
      endforeach;
   
      /**
       * Drop deprecated indexes
       */
      $indexList = $setup->getConnection()->getIndexList('totem_menumanager_node');
      foreach($indexList as $indexData):
         if(count($indexData['COLUMNS_LIST']) != 1 || !in_array($indexData['COLUMNS_LIST'][0], ['cms_page_identifier', 'category_id'])):
            continue;
         endif;
      
         $setup->getConnection()->dropIndex(
            $tableNode,
            $indexData['KEY_NAME']
         );
      endforeach;
   
      /**
       * Modify deprecated columns
       */
      $setup->getConnection()->modifyColumn(
         $tableNode,
         'cms_page_identifier',
         [
            'type'      => Table::TYPE_TEXT,
            'length'    => 100,
            'comment'   => 'DEPRECATED: Page String Identifier'
         ]
      );

      $setup->getConnection()->modifyColumn(
         $tableNode,
         'category_id',
         [
            'type'      => Table::TYPE_INTEGER,
            'length'    => 10,
            'comment'   => 'DEPRECATED: Category ID'
         ]
      );
      
      return $this;
   }
}
