<?php
/**
 * @author     Bart de Groot
 * @company    Abovo Media
 * @copyright  Copyright (c) 2019 Abovo Media (http://www.abovomedia.nl)
 * @package    Totem_MenuManager
 */

namespace Totem\MenuManager\Model;

use Totem\MenuManager\Api\MenuRepositoryInterface;
use Totem\MenuManager\Api\Data\MenuInterface;
use Totem\MenuManager\Model\ResourceModel\Menu\CollectionFactory;
use Totem\MenuManager\Api\Data\MenuSearchResultsInterfaceFactory;

use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\CouldNotDeleteException;

class MenuRepository implements MenuRepositoryInterface
{
   /**
    * @var MenuFactory
    */
   protected $menuFactory;
   
   /**
    * @var CollectionFactory
    */
   protected $collectionFactory;
   
   /**
    * @var MenuSearchResultsInterfaceFactory
    */
   private $menuSearchResultsFactory;
   
   /**
    * @param MenuFactory                       $menuFactory
    * @param CollectionFactory                 $collectionFactory
    * @param MenuSearchResultsInterfaceFactory $menuSearchResults
    */
   public function __construct(
      MenuFactory $menuFactory,
      CollectionFactory $collectionFactory,
      MenuSearchResultsInterfaceFactory $menuSearchResults
   ) {
      $this->menuFactory               = $menuFactory;
      $this->collectionFactory         = $collectionFactory;
      $this->menuSearchResultsFactory  = $menuSearchResults;
   }
   
   /**
    * @param MenuInterface $object
    *
    * @return MenuInterface
    * @throws CouldNotSaveException
    */
   public function save(MenuInterface $object)
   {
      try
      {
         $object->save();
      }
      catch (Exception $e)
      {
         throw new CouldNotSaveException($e->getMessage());
      }
      
      return $object;
   }
   
   /**
    * @param int $id
    *
    * @return Menu
    * @throws NoSuchEntityException
    */
   public function getById($id)
   {
      $object = $this->menuFactory->create();
      $object->load($id);
      if (!$object->getId()):
         throw new NoSuchEntityException(__('Object with id "%1" does not exist.', $id));
      endif;
      
      return $object;
   }
   
   /**
    * @param MenuInterface $object
    *
    * @return bool
    * @throws CouldNotDeleteException
    */
   public function delete(MenuInterface $object)
   {
      try
      {
         $object->delete();
      }
      catch (Exception $exception)
      {
         throw new CouldNotDeleteException(__($exception->getMessage()));
      }
      
      return true;
   }
   
   /**
    * @param int $id
    *
    * @return bool
    * @throws CouldNotDeleteException
    * @throws NoSuchEntityException
    */
   public function deleteById($id)
   {
      return $this->delete($this->getById($id));
   }
   
   /**
    * @inheritdoc
    */
   public function getList(SearchCriteriaInterface $criteria)
   {
      $searchResults = $this->menuSearchResultsFactory->create();
      $searchResults->setSearchCriteria($criteria);
      $collection    = $this->collectionFactory->create();
      foreach ($criteria->getFilterGroups() as $filterGroup):
         $fields     = [];
         $conditions = [];
         foreach ($filterGroup->getFilters() as $filter):
            $condition     = $filter->getConditionType() ? $filter->getConditionType() : 'eq';
            $fields[]      = $filter->getField();
            $conditions[]  = [$condition => $filter->getValue()];
         endforeach;
         
         if ($fields):
            $collection->addFieldToFilter($fields, $conditions);
         endif;
      endforeach;
      
      $searchResults->setTotalCount($collection->getSize());
      $sortOrders = $criteria->getSortOrders();
      if ($sortOrders):
         /** @var SortOrder $sortOrder */
         foreach ($sortOrders as $sortOrder):
            $collection->addOrder(
               $sortOrder->getField(),
               ($sortOrder->getDirection() == SortOrder::SORT_ASC) ? 'ASC' : 'DESC'
            );
         endforeach;
      endif;
      
      $collection->setCurPage($criteria->getCurrentPage());
      $collection->setPageSize($criteria->getPageSize());
      
      $objects = [];
      foreach ($collection as $objectModel):
         $objects[] = $objectModel;
      endforeach;
      
      $searchResults->setItems($objects);
      
      return $searchResults;
   }
   
   /**
    * @param string $identifier
    * @param array  $storeIds
    *
    * @return Menu
    */
   public function get($identifier, $storeIds)
   {
      $collection = $this->collectionFactory->create();
      $collection->addFilter('identifier', $identifier);
      $collection->addFilter('is_active', 1);
      $collection->join(['stores' => 'totem_menumanager_store'], 'main_table.menu_id = stores.menu_id', 'store_id');
      $collection->addFieldToFilter('store_id', ['in' => $storeIds]);
      
      return $collection->getFirstItem();
   }
}
