<?php
/**
 * @author     Bart de Groot
 * @company    Abovo Media
 * @copyright  Copyright (c) 2019 Abovo Media (http://www.abovomedia.nl)
 * @package    Totem_MenuManager
 */

namespace Totem\MenuManager\Model\Menu;

use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Api\SearchResultsInterfaceFactory;
use Magento\Framework\Model\ResourceModel\Db\Collection\AbstractCollection;
use Totem\MenuManager\Api\Data\NodeInterface;
use Totem\MenuManager\Api\NodeRepositoryInterface;
use Totem\MenuManager\Model\Menu\NodeFactory;
use Totem\MenuManager\Model\ResourceModel\Menu\Node\CollectionFactory;

class NodeRepository implements NodeRepositoryInterface
{
   /**
    * @var \Totem\MenuManager\Model\Menu\NodeFactory
    */
   protected $nodeFactory;
   
   /**
    * @var CollectionFactory
    */
   protected $collectionFactory;
   
   /**
    * @param \Totem\MenuManager\Model\Menu\NodeFactory $nodeFactory
    * @param CollectionFactory                         $collectionFactory
    * @param SearchResultsInterfaceFactory             $searchResultsFactory
    */
   public function __construct(
      NodeFactory $nodeFactory,
      CollectionFactory $collectionFactory,
      SearchResultsInterfaceFactory $searchResultsFactory
   ) {
      $this->nodeFactory            = $nodeFactory;
      $this->collectionFactory      = $collectionFactory;
      $this->searchResultsFactory   = $searchResultsFactory;
   }
   
   /**
    * @param NodeInterface $object
    *
    * @return NodeInterface
    * @throws CouldNotSaveException
    */
   public function save(NodeInterface $object)
   {
      try
      {
         $object->save();
      }
      catch (\Exception $e)
      {
         throw new CouldNotSaveException($e->getMessage());
      }
      
      return $object;
   }
   
   /**
    * @param int $id
    *
    * @return Node
    * @throws NoSuchEntityException
    */
   public function getById($id)
   {
      $object = $this->nodeFactory->create();
      $object->load($id);
      if (!$object->getId()):
         throw new NoSuchEntityException(__('Object with id "%1" does not exist.', $id));
      endif;
      
      return $object;
   }
   
   /**
    * @param NodeInterface $object
    *
    * @return bool
    * @throws CouldNotDeleteException
    */
   public function delete(NodeInterface $object)
   {
      try
      {
         $object->delete();
      }
      catch (\Exception $exception)
      {
         throw new CouldNotDeleteException(__($exception->getMessage()));
      }
      
      return true;
   }
   
   /**
    * @param int $id
    *
    * @return bool
    * @throws CouldNotDeleteException
    * @throws NoSuchEntityException
    */
   public function deleteById($id)
   {
      return $this->delete($this->getById($id));
   }
   
   /**
    * @param SearchCriteriaInterface $criteria
    *
    * @return \Magento\Framework\Api\SearchResultsInterface
    */
   public function getList(SearchCriteriaInterface $criteria)
   {
      $searchResults = $this->searchResultsFactory->create();
      $searchResults->setSearchCriteria($criteria);
      $collection    = $this->collectionFactory->create();
      foreach ($criteria->getFilterGroups() as $filterGroup):
         $fields     = [];
         $conditions = [];
         foreach ($filterGroup->getFilters() as $filter):
            $condition     = $filter->getConditionType() ? $filter->getConditionType() : 'eq';
            $fields[]      = $filter->getField();
            $conditions[]  = [$condition => $filter->getValue()];
         endforeach;
         
         if ($fields):
            $collection->addFieldToFilter($fields, $conditions);
         endif;
      endforeach;
      
      $searchResults->setTotalCount($collection->getSize());
      $sortOrders = $criteria->getSortOrders();
      if ($sortOrders):
         /** @var SortOrder $sortOrder */
         foreach ($sortOrders as $sortOrder):
            $collection->addOrder(
               $sortOrder->getField(),
               ($sortOrder->getDirection() == SortOrder::SORT_ASC) ? 'ASC' : 'DESC'
            );
         endforeach;
      endif;
      
      $collection->setCurPage($criteria->getCurrentPage());
      $collection->setPageSize($criteria->getPageSize());
      
      $objects = [];
      foreach ($collection as $objectModel):
         $objects[] = $objectModel;
      endforeach;
      
      $searchResults->setItems($objects);
      
      return $searchResults;
   }
   
   /**
    * @param int $menuId
    *
    * @return NodeInterface[]
    */
   public function getByMenu($menuId)
   {
      $collection = $this->collectionFactory->create();
      $collection->addFilter('menu_id', $menuId);
      $collection->addFilter('is_active', 1);
      $collection->addOrder('level', AbstractCollection::SORT_ORDER_ASC);
      $collection->addOrder('parent_id', AbstractCollection::SORT_ORDER_ASC);
      $collection->addOrder('position', AbstractCollection::SORT_ORDER_ASC);
      
      return $collection->getItems();
   }
}
