<?php
/**
 * @author     Bart de Groot
 * @company    Abovo Media
 * @copyright  Copyright (c) 2019 Abovo Media (http://www.abovomedia.nl)
 * @package    Totem_MenuManager
 */

namespace Totem\MenuManager\Model;

use Magento\Framework\DataObject\IdentityInterface;
use Magento\Framework\Model\AbstractModel;
use Totem\MenuManager\Api\Data\MenuInterface;

class Menu extends AbstractModel implements MenuInterface, IdentityInterface
{
   /**
    * Menu's Statuses
    */
   const STATUS_ENABLED = 1;
   const STATUS_DISABLED = 0;
   
   /**
    * Menu cache tag
    */
   const CACHE_TAG = 'totem_menumanager_menu';
   
   /**
    * @var string
    */
   protected $_cacheTag = self::CACHE_TAG;
   
   /**
    * Initialize resource model
    *
    * @return void
    */
   protected function _construct()
   {
      $this->_init(\Totem\MenuManager\Model\ResourceModel\Menu::class);
   }
   
   /**
    * Prepare menu's statuses.
    *
    * @return array
    */
   public function getAvailableStatuses()
   {
      return [self::STATUS_ENABLED => __('Enabled'), self::STATUS_DISABLED => __('Disabled')];
   }
   
   /**
    * Get identities
    *
    * @return array
    */
   public function getIdentities()
   {
      return [self::CACHE_TAG . '_' . $this->getId()];
   }
   
   /**
    * Receive menu store ids
    *
    * @return int[]
    */
   public function getStores()
   {
      $connection = $this->getResource()->getConnection();
      $select     = $connection->select()->from($this->getResource()->getTable('totem_menumanager_store'), ['store_id'])->where(
         'menu_id = ?',
         $this->getId()
      );
      
      return $connection->fetchCol($select);
   }
   
   /**
    * Save menu store ids
    *
    * @param array $stores
    */
   public function saveStores(array $stores)
   {
      $table      = $this->getResource()->getTable('totem_menumanager_store');
      $connection = $this->getResource()->getConnection();
      $connection->beginTransaction();
      $connection->delete($table, ['menu_id = ?' => $this->getId()]);
      
      foreach ($stores as $store):
         $connection->insert($table, ['menu_id' => $this->getId(), 'store_id' => $store]);
      endforeach;
      
      $connection->commit();
   }
   
   /**
    * @inheritdoc
    */
   public function getMenuId()
   {
      return $this->_getData(MenuInterface::MENU_ID);
   }
   
   /**
    * @inheritdoc
    */
   public function setMenuId($menuId)
   {
      return $this->setData(MenuInterface::MENU_ID, $menuId);
   }
   
   /**
    * @inheritdoc
    */
   public function getTitle()
   {
      return $this->_getData(MenuInterface::TITLE);
   }
   
   /**
    * @inheritdoc
    */
   public function setTitle($title)
   {
      return $this->setData(MenuInterface::TITLE, $title);
   }
   
   /**
    * @inheritdoc
    */
   public function getIdentifier()
   {
      return $this->_getData(MenuInterface::IDENTIFIER);
   }
   
   /**
    * @inheritdoc
    */
   public function setIdentifier($identifier)
   {
      return $this->setData(MenuInterface::IDENTIFIER, $identifier);
   }
   
   /**
    * @inheritdoc
    */
   public function getCreationTime()
   {
      return $this->_getData(MenuInterface::CREATION_TIME);
   }
   
   /**
    * @inheritdoc
    */
   public function setCssClass($cssClass)
   {
      return $this->setData(MenuInterface::CSS_CLASS, $cssClass);
   }
   
   /**
    * @inheritdoc
    */
   public function getCssClass()
   {
      return $this->_getData(MenuInterface::CSS_CLASS);
   }
   
   /**
    * @inheritdoc
    */
   public function setCreationTime($creationTime)
   {
      return $this->setData(MenuInterface::CREATION_TIME, $creationTime);
   }
   
   /**
    * @inheritdoc
    */
   public function getUpdateTime()
   {
      return $this->_getData(MenuInterface::UPDATE_TIME);
   }
   
   /**
    * @inheritdoc
    */
   public function setUpdateTime($updateTime)
   {
      return $this->setData(MenuInterface::UPDATE_TIME, $updateTime);
   }
   
   /**
    * @inheritdoc
    */
   public function getIsActive()
   {
      return $this->_getData(MenuInterface::IS_ACTIVE);
   }
   
   /**
    * @inheritdoc
    */
   public function setIsActive($isActive)
   {
      return $this->setData(MenuInterface::IS_ACTIVE, $isActive);
   }
}
