<?php
/**
 * @author     Bart de Groot
 * @company    Abovo Media
 * @copyright  Copyright (c) 2019 Abovo Media (http://www.abovomedia.nl)
 * @package    Totem_MenuManager
 */

namespace Totem\MenuManager\Controller\Adminhtml\Menu;

use Magento\Backend\App\Action;
use Magento\Catalog\Model\ProductRepository;
use Magento\Framework\Api\FilterBuilderFactory;
use Magento\Framework\Api\Search\FilterGroupBuilderFactory;
use Magento\Framework\Api\SearchCriteriaBuilderFactory;
use Magento\Framework\App\ResponseInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Totem\MenuManager\Api\MenuRepositoryInterface;
use Totem\MenuManager\Api\NodeRepositoryInterface;
use Totem\MenuManager\Model\Menu\NodeFactory;
use Totem\MenuManager\Model\MenuFactory;

class Save extends Action
{
   const ADMIN_RESOURCE = 'Totem_MenuManager::menumanager';
   
   /**
    * @var MenuRepositoryInterface
    */
   private $menuRepository;
   
   /**
    * @var NodeRepositoryInterface
    */
   private $nodeRepository;
   
   /**
    * @var FilterBuilderFactory
    */
   private $filterBuilderFactory;
   
   /**
    * @var FilterGroupBuilderFactory
    */
   private $filterGroupBuilderFactory;
   
   /**
    * @var SearchCriteriaBuilderFactory
    */
   private $searchCriteriaBuilderFactory;
   
   /**
    * @var NodeFactory
    */
   private $nodeFactory;
   
   /**
    * @var MenuFactory
    */
   private $menuFactory;
   
   /**
    * @var ProductRepository
    */
   private $productRepository;
   
   /**
    * @param Action\Context               $context
    * @param MenuRepositoryInterface      $menuRepository
    * @param NodeRepositoryInterface      $nodeRepository
    * @param FilterBuilderFactory         $filterBuilderFactory
    * @param FilterGroupBuilderFactory    $filterGroupBuilderFactory
    * @param SearchCriteriaBuilderFactory $searchCriteriaBuilderFactory
    * @param NodeFactory                  $nodeFactory
    * @param MenuFactory                  $menuFactory
    * @param ProductRepository            $productRepository
    */
   public function __construct(
      Action\Context $context,
      MenuRepositoryInterface $menuRepository,
      NodeRepositoryInterface $nodeRepository,
      FilterBuilderFactory $filterBuilderFactory,
      FilterGroupBuilderFactory $filterGroupBuilderFactory,
      SearchCriteriaBuilderFactory $searchCriteriaBuilderFactory,
      NodeFactory $nodeFactory,
      MenuFactory $menuFactory,
      ProductRepository $productRepository
   ) {
      parent::__construct($context);
      
      $this->menuRepository               = $menuRepository;
      $this->nodeRepository               = $nodeRepository;
      $this->filterBuilderFactory         = $filterBuilderFactory;
      $this->filterGroupBuilderFactory    = $filterGroupBuilderFactory;
      $this->searchCriteriaBuilderFactory = $searchCriteriaBuilderFactory;
      $this->nodeFactory                  = $nodeFactory;
      $this->menuFactory                  = $menuFactory;
      $this->productRepository            = $productRepository;
   }
   
   /**
    * Dispatch request
    *
    * @return \Magento\Framework\Controller\ResultInterface|ResponseInterface
    * @throws \Magento\Framework\Exception\NotFoundException
    */
   public function execute()
   {
      $data = $this->getRequest()->getPostValue('menu');
      
      if(isset($data['menu_id']) && ($id = $data['menu_id'])):
         $menu = $this->menuRepository->getById($id);
      else:
         $menu = $this->menuFactory->create();
      endif;
      
      $menu->setTitle($data['title']);
      $menu->setIdentifier($data['identifier']);
      $menu->setCssClass($data['css_class']);
      $menu->setIsActive($data['is_active']);
   
      try
      {
         $menu = $this->menuRepository->save($menu);
         
         $this->messageManager->addSuccessMessage(__('You saved this menu.'));
      
      }
      catch (\RuntimeException $e)
      {
         $this->messageManager->addErrorMessage($e->getMessage());
      }
      catch (\Exception $e)
      {
         $this->messageManager->addExceptionMessage($e, __('Something went wrong while saving menu.'));
      }
      
      if (!isset($id)):
         $id = $menu->getId();
      endif;
      
      $menu->saveStores($data['stores']);
      
      $nodes = $this->getRequest()->getPostValue('serialized_nodes');
      if (!empty($nodes)):
         $nodes = json_decode($nodes, true);
         $nodes = $this->_convertTree($nodes, '#');
         
         if (!empty($nodes)):
            $filterBuilder          = $this->filterBuilderFactory->create();
            $filter                 = $filterBuilder->setField('menu_id')->setValue($id)->setConditionType('eq')->create();
            
            $filterGroupBuilder     = $this->filterGroupBuilderFactory->create();
            $filterGroup            = $filterGroupBuilder->addFilter($filter)->create();
            
            $searchCriteriaBuilder  = $this->searchCriteriaBuilderFactory->create();
            $searchCriteria         = $searchCriteriaBuilder->setFilterGroups([$filterGroup])->create();
            
            $oldNodes               = $this->nodeRepository->getList($searchCriteria)->getItems();
            
            $existingNodes = [];
            foreach ($oldNodes as $node):
               $existingNodes[$node->getId()] = $node;
            endforeach;
            
            $nodesToDelete = [];
            foreach ($existingNodes as $nodeId => $noe):
               $nodesToDelete[$nodeId] = true;
            endforeach;
            
            $nodeMap = [];
            foreach ($nodes as $node):
               $nodeId  = $node['id'];
               $matches = [];
               if (preg_match('/^node_([0-9]+)$/', $nodeId, $matches)):
                  $nodeId = $matches[1];
                  unset($nodesToDelete[$nodeId]);
                  $nodeMap[$node['id']] = $existingNodes[$nodeId];
               else:
                  $nodeObject = $this->nodeFactory->create();
                  $nodeObject->setMenuId($id);
                  $nodeObject = $this->nodeRepository->save($nodeObject);
                  $nodeMap[$nodeId] = $nodeObject;
               endif;
            endforeach;
            
            foreach (array_keys($nodesToDelete) as $nodeId):
               $this->nodeRepository->deleteById($nodeId);
            endforeach;
            
            $path = [
               '#' => 0,
            ];
            foreach ($nodes as $node):
               if ($node['type'] == 'product' && !$this->validateProductNode($node)):
                  continue;
               endif;
               
               $nodeObject = $nodeMap[$node['id']];
               
               $parents    = array_keys($path);
               $parent     = array_pop($parents);
               while ($parent != $node['parent']):
                  array_pop($path);
                  $parent = array_pop($parents);
               endwhile;
               
               $level      = count($path) - 1;
               $position   = $path[$node['parent']]++;
               
               if ($node['parent'] == '#'):
                  $nodeObject->setParentId(null);
               else:
                  $nodeObject->setParentId($nodeMap[$node['parent']]->getId());
               endif;
               
               $nodeObject->setType($node['type']);
               if (isset($node['classes'])):
                  $nodeObject->setClasses($node['classes']);
               endif;
               if (isset($node['button'])):
                  $nodeObject->setButton($node['button']);
               endif;
               if (isset($node['align'])):
                  $nodeObject->setAlign($node['align']);
               endif;
               if (isset($node['content'])):
                  $nodeObject->setContent($node['content']);
               endif;
               if (isset($node['target'])):
                  $nodeObject->setTarget($node['target']);
               endif;
               if (isset($node['subcategories'])):
                  $nodeObject->setSubcategories($node['subcategories']);
               endif;
               
               $nodeObject->setMenuId($id);
               $nodeObject->setTitle($node['title']);
               $nodeObject->setIsActive(1);
               $nodeObject->setLevel($level);
               $nodeObject->setPosition($position);
               
               $this->nodeRepository->save($nodeObject);
               
               $path[$node['id']] = 0;
            endforeach;
         endif;
      endif;
      
      $redirect = $this->resultRedirectFactory->create();
      $redirect->setPath('*/*/index');
      
      if ($this->getRequest()->getParam('back')):
         $redirect->setPath('*/*/edit', ['id' => $menu->getId(), '_current' => true]);
      endif;
      
      return $redirect;
   }
   
   /**
    * @param $nodes
    * @param $parent
    *
    * @return array
    */
   protected function _convertTree($nodes, $parent)
   {
      $convertedTree = [];
      if (!empty($nodes)):
         foreach ($nodes as $node):
            $node['parent']   = $parent;
            $convertedTree[]  = $node;
            $convertedTree    = array_merge($convertedTree, $this->_convertTree($node['columns'], $node['id']));
         endforeach;
      endif;
      
      return $convertedTree;
   }
   
   /**
    * @param array $node
    * @return bool
    */
   private function validateProductNode(array $node)
   {
      try
      {
         $this->productRepository->getById($node['content']);
      }
      catch (NoSuchEntityException $e)
      {
         $this->messageManager->addErrorMessage(__('Product does not exist'));
         
         return false;
      }
      
      return true;
   }
}
