<?php
/**
 * @author     Bart de Groot
 * @company    Abovo Media
 * @copyright  Copyright (c) 2019 Abovo Media (http://www.abovomedia.nl)
 * @package    Totem_MenuManager
 */

namespace Totem\MenuManager\Block;

use Magento\Framework\App\Cache\Type\Block;
use Magento\Framework\DataObject;
use Magento\Framework\View\Element\Template;
use Magento\Framework\Event\Manager as EventManager;
use Magento\Catalog\Api\CategoryRepositoryInterface;
use Magento\Catalog\Model\ResourceModel\Category\CollectionFactory as CategoryCollection;
use Totem\MenuManager\Api\MenuRepositoryInterface;
use Totem\MenuManager\Api\NodeRepositoryInterface;
use Totem\MenuManager\Model\NodeTypeProvider;
use Totem\MenuManager\Model\TemplateResolver;
use Totem\MenuManager\Model\Menu\NodeFactory;
use Totem\MenuManager\Block\NodeType\Category as NodeCategory;
use Totem\MenuManager\Block\NodeType\CmsPage as NodeCmsPage;
use Totem\MenuManager\Block\NodeType\Product as NodeProduct;
use Totem\MenuManager\Block\NodeType\CustomUrl as NodeCustomUrl;

class Menu extends Template implements DataObject\IdentityInterface
{
   public $nodesActive     = [];
   
   private $menu           = [];
   private $nodes;
   private $recursionLevel;
   
   private $nodesFlat      = [];
   
   /**
    * @var MenuRepositoryInterface
    */
   private $menuRepository;
   
   /**
    * @var NodeRepositoryInterface
    */
   private $nodeRepository;
   
   /**
    * @var NodeTypeProvider
    */
   private $nodeTypeProvider;
   
   /**
    * @var EventManager
    */
   private $eventManager;
   
   /**
    * @var TemplateResolver
    */
   private $templateResolver;
   
   /**
    * @var CategoryRepositoryInterface
    */
   private $categoryRepository;
   
   /**
    * @var CategoryCollection
    */
   private $categoryCollection;
   
   /**
    * @var NodeFactory
    */
   private $nodeFactory;
   
   /**
    * @var string
    */
   protected $submenuTemplate;
   
   /**
    * @var NodeCategory
    */
   private $nodeCategory;
   
   /**
    * @var NodeCmsPage
    */
   private $nodeCmsPage;
   
   /**
    * @var NodeProduct
    */
   private $nodeProduct;
   
   /**
    * @var NodeCustomUrl
    */
   private $nodeCustomUrl;
   
   /**
    * @var string
    */
   protected $_template = 'Totem_MenuManager::html/menu.phtml';
   
   public function __construct(
      Template\Context $context,
      EventManager $eventManager,
      MenuRepositoryInterface $menuRepository,
      NodeRepositoryInterface $nodeRepository,
      NodeTypeProvider $nodeTypeProvider,
      TemplateResolver $templateResolver,
      CategoryRepositoryInterface $categoryRepository,
      CategoryCollection $categoryCollection,
      NodeFactory $nodeFactory,
      NodeCategory $nodeCategory,
      NodeCmsPage $nodeCmsPage,
      NodeProduct $nodeProduct,
      NodeCustomUrl $nodeCustomUrl,
      array $data = []
   ) {
      parent::__construct($context, $data);
      
      $this->eventManager        = $eventManager;
      $this->menuRepository      = $menuRepository;
      $this->nodeRepository      = $nodeRepository;
      $this->nodeTypeProvider    = $nodeTypeProvider;
      $this->templateResolver    = $templateResolver;
      $this->categoryRepository  = $categoryRepository;
      $this->categoryCollection  = $categoryCollection;
      $this->nodeFactory         = $nodeFactory;
      $this->nodeCategory        = $nodeCategory;
      $this->nodeCmsPage         = $nodeCmsPage;
      $this->nodeProduct         = $nodeProduct;
      $this->nodeCustomUrl       = $nodeCustomUrl;
      $this->submenuTemplate     = $this->getMenuTemplate(
         'Totem_MenuManager::html/menu/sub_menu.phtml'
      );
      $this->setTemplate($this->getMenuTemplate($this->_template));
      
      $this->recursionLevel = max(
         0,
         (int)$this->getConfig('totem_menumanager/settings/max_depth')
      );
   }
   
   /**
    * Return unique ID(s) for each object in system
    *
    * @return string[]
    */
   public function getIdentities()
   {
      return [\Totem\MenuManager\Model\Menu::CACHE_TAG, Block::CACHE_TAG];
   }
   
   /**
    * @return bool|float|int|null
    */
   protected function getCacheLifetime()
   {
      return 60*60*24*365;
   }
   
   /**
    * @param $configPath
    *
    * @return mixed
    */
   public function getConfig($configPath)
   {
      return $this->_scopeConfig->getValue(
         $configPath,
         \Magento\Store\Model\ScopeInterface::SCOPE_STORE
      );
   }
   
   /**
    * @return \Totem\MenuManager\Model\Menu
    */
   private function loadMenu()
   {
      $identifier = $this->getData('identifier');
      
      if(!array_key_exists($identifier, $this->menu)):
         $this->menu[$identifier] = $this->menuRepository->get($identifier, $this->_getStoresToFilter());
      endif;
      
      return $this->menu[$identifier];
   }
   
   /**
    * @return \Totem\MenuManager\Model\Menu|null
    */
   public function getMenu()
   {
      $menu = $this->loadMenu();
      if (!$menu->getMenuId()):
         return null;
      endif;
      
      return $menu;
   }
   
   /**
    * @return array
    * @throws \Magento\Framework\Exception\NoSuchEntityException
    */
   public function getCacheKeyInfo()
   {
      $info = [
         \Totem\MenuManager\Model\Menu::CACHE_TAG,
         'menu_' . $this->loadMenu()->getId(),
         'store_' . $this->_storeManager->getStore()->getId(),
         'template_' . $this->getTemplate()
      ];
      
      $nodeCacheKeyInfo = $this->getNodeCacheKeyInfo();
      if ($nodeCacheKeyInfo):
         $info = array_merge($info, $nodeCacheKeyInfo);
      endif;
      
      return $info;
   }
   
   /**
    * @return array
    */
   private function getNodeCacheKeyInfo()
   {
      $info       = [];
      $nodeType   = '';
      $request    = $this->getRequest();
      
      switch ($request->getRouteName()):
         case 'cms':
            $nodeType = 'cms_page';
            break;
         case 'catalog':
            $nodeType = 'category';
            break;
      endswitch;
      
      $transport = [
         'node_type' => $nodeType,
         'request'   => $request
      ];
      
      $transport = new DataObject($transport);
      $this->eventManager->dispatch(
         'totem_menumanager_cache_node_type',
         ['transport' => $transport]
      );
      
      if ($transport->getNodeType()):
         $nodeType = $transport->getNodeType();
      endif;
      
      if ($nodeType):
         $info = $this->getNodeTypeProvider($nodeType)->getNodeCacheKeyInfo();
      endif;
      
      if ($this->getParentNode()):
         $info[] = 'parent_node_' . $this->getParentNode()->getNodeId();
      endif;
      
      return $info;
   }
   
   /**
    * @param string $nodeType
    * @return bool
    */
   public function isViewAllLinkAllowed($nodeType)
   {
      return $this->getNodeTypeProvider($nodeType)->isViewAllLinkAllowed();
   }
   
   /**
    * @param NodeRepositoryInterface $node
    * @return string
    */
   public function renderViewAllLink($node)
   {
      return $this->getMenuNodeBlock($node)
                  ->setIsViewAllLink(true)
                  ->toHtml();
   }
   
   /**
    * @param NodeRepositoryInterface $node
    * @return string
    */
   public function renderMenuNode($node)
   {
      return $this->getMenuNodeBlock($node)->toHtml();
   }
   
   /**
    * @param array $nodes
    * @param NodeRepositoryInterface $parentNode
    * @param int $level
    * @return string
    */
   public function renderSubmenu($nodes, $parentNode, $level = 0)
   {
      return $nodes
         ? $this->getSubmenuBlock($nodes, $parentNode, $level)->toHtml()
         : '';
   }
   
   /**
    * @param int $level
    * @param NodeRepositoryInterface|null $parent
    * @return array
    */
   public function getNodesTree($level = 0, $parent = null)
   {
      $nodesTree  = [];
      $nodes      = $this->getNodes($level, $parent);
      
      foreach ($nodes as $node):
         $nodesTree[] = [
            'node'      => $node,
            'children'  => $this->getNodesTree($level + 1, $node)
         ];
      endforeach;
      
      return $nodesTree;
   }
   
   /**
    * @param string $nodeType
    * @return \Totem\MenuManager\Api\NodeTypeInterface
    */
   public function getNodeTypeProvider($nodeType)
   {
      return $this->nodeTypeProvider->getProvider($nodeType);
   }
   
   /**
    * @param int  $level
    * @param null $parent
    *
    * @return array
    */
   public function getNodes($level = 0, $parent = null)
   {
      if (empty($this->nodes)):
         $this->fetchData();
      endif;
      
      if (!isset($this->nodes[$level])):
         return [];
      endif;
      
      $parentId = $parent['node_id'] ?: 0;
      if (!isset($this->nodes[$level][$parentId])):
         return [];
      endif;
      
      return $this->nodes[$level][$parentId];
   }
   
   /**
    * Builds HTML tag attributes from an array of attributes data
    *
    * @param array $array
    * @return string
    */
   public function buildAttrFromArray(array $array)
   {
      $attributes = [];
      
      foreach ($array as $attribute => $data):
         if (is_array($data)):
            $data = implode(' ', $data);
         endif;
         
         $attributes[] = $attribute . '="' . htmlspecialchars($data) . '"';
      endforeach;
      
      return $attributes ? ' ' . implode(' ', $attributes) : '';
   }
   
   /**
    * @param string $defaultClass
    * @return string
    */
   public function getMenuCssClass($defaultClass = '')
   {
      $menu = $this->getMenu();
      
      if (is_null($menu)):
         return $defaultClass;
      endif;
      
      return $menu->getCssClass();
   }
   
   /**
    * @param NodeRepositoryInterface $node
    * @return Template
    */
   private function getMenuNodeBlock($node)
   {
      $nodeBlock  = $this->getNodeTypeProvider($node->getType());
      
      $level      = $node->getLevel();
      $isRoot     = 0 == $level;
      $nodeBlock->setId($node->getNodeId())
                ->setTitle($node->getTitle())
                ->setLevel($level)
                ->setIsRoot($isRoot)
                ->setIsParent((bool) $node->getIsParent())
                ->setIsViewAllLink(false)
                ->setContent($node->getContent())
                ->setNodeClasses($node->getClasses())
                ->setButton($node->getButton())
                ->setAlign($node->getAlign())
                ->setMenuClass($this->getMenu()->getCssClass())
                ->setMenuCode($this->getData('identifier'))
                ->setTarget($node->getTarget());
      
      return $nodeBlock;
   }
   
   /**
    * @param array $nodes
    * @param NodeRepositoryInterface $parentNode
    * @param int $level
    * @return Menu
    */
   private function getSubmenuBlock($nodes, $parentNode, $level = 0)
   {
      $block = clone $this;
      
      $block->setSubmenuNodes($nodes)
            ->setParentNode($parentNode)
            ->setLevel($level);
      
      $block->setTemplateContext($block);
      $block->setTemplate($this->submenuTemplate);
      
      return $block;
   }
   
   private function fetchData()
   {
      $nodes   = $this->nodeRepository->getByMenu($this->loadMenu()->getId());
      $result  = [];
      $types   = [];
      
      foreach ($nodes as $node):
         if($node->getType() != 'category' || !$node->getSubcategories()):
            continue;
         endif;
         
         $subcategories = [];
         $children      = $this->_getSubCategories($node, $subcategories);
         $nodes         = array_merge($nodes, $children);
      endforeach;
      
      foreach ($nodes as $node):
         $this->nodesFlat[$node->getNodeId()] = $node;
         
         $level = $node->getLevel();
         if($level > $this->recursionLevel):
            continue;
         endif;
         
         $parent = $node->getParentId() ?: 0;
         if (!isset($result[$level])):
            $result[$level] = [];
         endif;
         if (!isset($result[$level][$parent])):
            $result[$level][$parent] = [];
         endif;
         
         $result[$level][$parent][] = $node;
         
         $type = $node->getType();
         if (!isset($types[$type])):
            $types[$type] = [];
         endif;
         
         $types[$type][] = $node;
      endforeach;
      
      $this->nodes = $result;
      
      foreach ($types as $type => $typeNodes):
         $this->nodeTypeProvider->prepareData($type, $typeNodes);
      endforeach;
      
      foreach ($nodes as $node):
         if(!$this->isCurrent($node) || !empty($this->nodesActive)):
            continue;
         endif;
         
         $this->getCurrentTree($node);
      endforeach;
      
      if(empty($this->nodesActive) && $this->getTemplate() == 'Totem_MenuManager::main_menu/html/menu.phtml'):
         reset($this->nodesFlat);
         $this->nodesActive[0] = key($this->nodesFlat);
      endif;
   }
   
   /**
    * Checks if menu item is current
    *
    * @param \Totem\MenuManager\Api\Data\NodeInterface $node
    * @return bool
    */
   public function isCurrent($node)
   {
      switch($node->getType()):
         case 'category':
            return $this->nodeCategory->isCurrentCategory($node->getNodeId());
            
            break;
         case 'cms_page':
            return $this->nodeCmsPage->isCurrentPage($node->getNodeId());
            
            break;
         case 'product':
            return $this->nodeProduct->isCurrentProduct($node->getNodeId());
            
            break;
         case 'custom_url':
            return $this->nodeCustomUrl->isCurrentUrl($node->getNodeId());
   
            break;
      endswitch;
      
      return false;
   }
   
   /**
    * Checks if menu item is current
    *
    * @param \Totem\MenuManager\Api\Data\NodeInterface $node
    * @return bool
    */
   public function getCurrentTree($node)
   {
      $this->nodesActive[$node->getLevel()] = $node->getNodeId();
      
      if($node->getParentId()):
         $this->getCurrentTree($this->nodesFlat[$node->getParentId()]);
      endif;
   }
   
   /**
    * @param string $template
    * @return string
    */
   protected function getMenuTemplate($template)
   {
      return $this->templateResolver->getMenuTemplate(
         $this,
         $this->getData('identifier'),
         $template
      );
   }
   
   /**
    * @return \Magento\Store\Api\Data\StoreInterface|null|string
    */
   private function _getStoresToFilter()
   {
      $storesToFilter   = [];
      $storesToFilter[] = 0; // all stores flag.
      $storesToFilter[] = $this->_storeManager->getStore()->getId();
      
      return $storesToFilter;
   }
   
   /**
    * Return child categories for category item
    *
    * @param \Magento\Framework\Data\Tree\Node $child
    *
    * @return array
    */
   private function _getSubCategories($node, &$subcategories)
   {
      $category = $this->_getCategory($node->getContent());
      
      if(!$category):
         return [];
      endif;
      
      foreach($this->_getChildrenCategories($category) as $category):
         if(!$category->getIsActive() || !$category->getIncludeInMenu()):
            continue;
         endif;
         
         $itemNode = $this->nodeFactory->create();
         $itemNode
            ->setNodeId($node->getNodeId() . '_cat_' . $category->getId())
            ->setMenuId($node->getMenuId())
            ->setType('category')
            ->setContent($category->getId())
            ->setClasses(null)
            ->setButton(null)
            ->setAlign(null)
            ->setParentId($node->getNodeId())
            ->setPosition($category->getPosition())
            ->setLevel($node->getLevel() + 1)
            ->setTitle($category->getName())
            ->setTarget(0)
            ->setSubcategories(1)
            ->setCreationTime(date('Y-m-d H:i:s'))
            ->setUpdateTime(date('Y-m-d H:i:s'))
            ->setIsActive($category->getIsActive());
         
         $subcategories[] = $itemNode;
         
         if($category->getChildren() && $itemNode->getLevel() < $this->recursionLevel):
            $this->_getSubCategories($itemNode, $subcategories);
         endif;
      endforeach;
      
      return $subcategories;
   }
   
   /**
    * Return child categories with include_in_menu
    *
    * @param \Magento\Catalog\Model\Category $category
    * @return \Magento\Catalog\Model\ResourceModel\Category\CollectionFactory
    */
   private function _getChildrenCategories($category)
   {
      $collection = $this->categoryCollection->create();
      $collection->addAttributeToSelect('url_key')
                 ->addAttributeToSelect('name')
                 ->addAttributeToSelect('all_children')
                 ->addAttributeToSelect('is_anchor')
                 ->addAttributeToSelect('include_in_menu')
                 ->addAttributeToFilter('is_active', 1)
                 ->addIdFilter($category->getChildren())
                 ->setOrder('position', \Magento\Framework\DB\Select::SQL_ASC)
                 ->addUrlRewriteToResult();
      
      return $collection;
   }
   
   /**
    * @param      $categoryId
    * @param null $storeId
    *
    * @return \Magento\Catalog\Api\Data\CategoryInterface
    * @throws \Magento\Framework\Exception\NoSuchEntityException
    */
   private function _getCategory($categoryId, $storeId = null)
   {
      try
      {
         return $this->categoryRepository->get($categoryId, $storeId);
      }
      catch(\Magento\Framework\Exception\NoSuchEntityException $e)
      {
         return false;
      }
   }
}
