<?php
/**
 * @author     Bart de Groot
 * @company    Abovo Media
 * @copyright  Copyright (c) 2018 Abovo Media (http://www.abovomedia.nl)
 * @package    Totem_Core
 */

namespace Totem\Core\Helper;

use Magento\Framework\App\Helper\AbstractHelper;
use SimpleXMLElement;
use Zend\Http\Client\Adapter\Curl as CurlClient;
use Zend\Http\Response as HttpResponse;
use Zend\Uri\Http as HttpUri;
use Magento\Framework\Serialize\Serializer\Json;

class Module extends AbstractHelper
{
   const MODULES_PATH = 'totem_modules';
   const URL_MODULES  = 'https://satis.abovo.nl/feed-modules-totem-mage2.xml';
   
   /**
    * @var CurlClient
    */
   protected $curlClient;
   
   /**
    * @var \Magento\Framework\App\CacheInterface
    */
   protected $cache;
   
   /**
    * @var \Magento\Framework\Module\Dir\Reader
    */
   private $_moduleReader;
   
   /**
    * @var \Magento\Framework\Filesystem\Driver\File
    */
   private $_filesystem;
   
   /**
    * @var Json
    */
   private $_jsonSerializer;
   
   public function __construct(
      \Magento\Framework\App\Helper\Context $context,
      \Magento\Framework\App\CacheInterface $cache,
      \Magento\Framework\Module\Dir\Reader $_moduleReader,
      \Magento\Framework\Filesystem\Driver\File $_filesystem,
      Json $_jsonSerializer,
      CurlClient $curl
   ) {
      parent::__construct($context);
      
      $this->cache            = $cache;
      $this->curlClient       = $curl;
      $this->_moduleReader    = $_moduleReader;
      $this->_filesystem      = $_filesystem;
      $this->_jsonSerializer  = $_jsonSerializer;
   }
   
   /**
    * Get array with info about all Totem Modules
    * @return bool|mixed
    */
   public function getAllModules()
   {
      $serialized = $this->cache->load(self::MODULES_PATH);
      if ($serialized === false):
         $this->reload();
         $serialized = $this->cache->load(self::MODULES_PATH);
      endif;
      
      $result = $this->_jsonSerializer->unserialize($serialized);
      
      return $result;
   }
   
   /**
    * Save modules data to magento cache
    */
   protected function reload()
   {
      $feedData   = [];
      $feedXml    = $this->getFeedData();
      if ($feedXml && $feedXml->channel && $feedXml->channel->item):
         foreach ($feedXml->channel->item as $item):
            $code = (string)$item->code;
            
            if (!isset($feedData[$code])):
               $feedData[$code] = [];
            endif;
            
            $feedData[$code][(string)$item->title] = [
               'name'      => (string)$item->title,
               'version'   => (string)$item->version
            ];
         endforeach;
         
         if ($feedData):
            $this->cache->save($this->_jsonSerializer->serialize($feedData), self::MODULES_PATH);
         endif;
      endif;
   }
   
   /**
    * Read data from xml file with curl
    *
    * @return bool|SimpleXMLElement
    */
   protected function getFeedData()
   {
      try
      {
         $curlClient = $this->getCurlClient();
         
         $location   = self::URL_MODULES;
         $uri        = new HttpUri($location);
         
         $curlClient->setOptions([
            'timeout'   => 8
         ]);
         
         $curlClient->connect($uri->getHost(), $uri->getPort());
         $curlClient->write('GET', $uri, 1.0);
         $data = HttpResponse::fromString($curlClient->read());
         
         $curlClient->close();
         
         $xml  = new SimpleXMLElement($data->getContent());
      }
      catch (\Exception $e)
      {
         return false;
      }
      
      return $xml;
   }
   
   /**
    * Returns the cURL client that is being used.
    *
    * @return CurlClient
    */
   public function getCurlClient()
   {
      if ($this->curlClient === null):
         $this->curlClient = new CurlClient();
      endif;
      
      return $this->curlClient;
   }
   
   /**
    * Read info about module from composer json file
    * @param $moduleCode
    * @return mixed
    * @throws \Magento\Framework\Exception\FileSystemException
    */
   public function getModuleInfo($moduleCode)
   {
      try
      {
         $dir     = $this->_moduleReader->getModuleDir('', $moduleCode);
         $file    = $dir . '/composer.json';
         
         $string  = $this->_filesystem->fileGetContents($file);
         $json    = $this->_jsonSerializer->unserialize($string);
      }
      catch (\Magento\Framework\Exception\FileSystemException $e)
      {
         $json = [];
      }
      
      return $json;
   }
}
