<?php
/**
 * @author     Bart de Groot
 * @company    Abovo Media
 * @copyright  Copyright (c) 2018 Abovo Media (http://www.abovomedia.nl)
 * @package    Totem_Core
 */

namespace Totem\Core\Block;

use Magento\Framework\Data\Form\Element\AbstractElement;

class Modules extends \Magento\Config\Block\System\Config\Form\Fieldset
{
   private $_fieldRenderer;
   
   /**
    * @var \Magento\Framework\Module\ModuleListInterface
    */
   protected $moduleList;
   
   /**
    * @var \Magento\Framework\View\LayoutFactory
    */
   protected $layoutFactory;
   
   /**
    * @var \Totem\Core\Helper\Module
    */
   protected $moduleHelper;
   
   /**
    * @param \Magento\Backend\Block\Context                $context
    * @param \Magento\Backend\Model\Auth\Session           $authSession
    * @param \Magento\Framework\View\Helper\Js             $jsHelper
    * @param \Magento\Framework\Module\ModuleListInterface $moduleList
    * @param \Magento\Framework\View\LayoutFactory         $layoutFactory
    * @param \Totem\Core\Helper\Module                     $moduleHelper
    * @param array                                         $data
    */
   public function __construct(
      \Magento\Backend\Block\Context $context,
      \Magento\Backend\Model\Auth\Session $authSession,
      \Magento\Framework\View\Helper\Js $jsHelper,
      \Magento\Framework\Module\ModuleListInterface $moduleList,
      \Magento\Framework\View\LayoutFactory $layoutFactory,
      \Totem\Core\Helper\Module $moduleHelper,
      array $data = []
   ) {
      parent::__construct($context, $authSession, $jsHelper, $data);
      
      $this->moduleList    = $moduleList;
      $this->layoutFactory = $layoutFactory;
      $this->moduleHelper  = $moduleHelper;
      $this->scopeConfig   = $context->getScopeConfig();
   }
   
   /**
    * Render fieldset html
    *
    * @param AbstractElement $element
    *
    * @return string
    */
   public function render(AbstractElement $element)
   {
      $html             = $this->_getHeaderHtml($element);
      
      $modules          = $this->moduleList->getNames();
      
      $dispatchResult   = new \Magento\Framework\DataObject($modules);
      $modules          = $dispatchResult->toArray();
      
      sort($modules);
      foreach ($modules as $moduleName):
         if (strstr($moduleName, 'Totem_') === false || $moduleName === 'Totem_Core'):
            continue;
         endif;
         
         $html .= $this->_getFieldHtml($element, $moduleName);
      endforeach;
      
      $html .= $this->_getFooterHtml($element);
      
      return $html;
   }
   
   /**
    * @return \Magento\Framework\View\Element\BlockInterface
    */
   protected function _getFieldRenderer()
   {
      if (empty($this->_fieldRenderer)):
         $layout = $this->layoutFactory->create();
         
         $this->_fieldRenderer = $layout->createBlock(
            \Magento\Config\Block\System\Config\Form\Field::class
         );
      endif;
      
      return $this->_fieldRenderer;
   }
   
   /**
    * Read info about modules from composer json file
    * @param $moduleCode
    * @return mixed
    * @throws \Magento\Framework\Exception\FileSystemException
    */
   protected function _getModuleInfo($moduleCode)
   {
      return $this->moduleHelper->getModuleInfo($moduleCode);
   }
   
   /**
    * @param $fieldset
    * @param $moduleCode
    *
    * @return string
    */
   protected function _getFieldHtml($fieldset, $moduleCode)
   {
      $module = $this->_getModuleInfo($moduleCode);
      if (!is_array($module) || !array_key_exists('version', $module) || !array_key_exists('description', $module)):
         return '';
      endif;
      
      $currentVer    = $module['version'];
      $moduleName    = $module['description'];
      $moduleName    = $this->_replaceTotemText($moduleName);
      $composerCode  = $module['name'];
      $status        ='<a target="_blank">
         <img src="'. $this->getViewFileUrl('Totem_Core::images/ok.gif') . '" title="' . __("Installed") . '" />
      </a>';
      
      $allModules    = $this->moduleHelper->getAllModules();
      if ($allModules && isset($allModules[$composerCode])):
         $singleRecord  = array_key_exists('name', $allModules[$composerCode]);
         $ext           = $singleRecord ? $allModules[$composerCode] : end($allModules[$composerCode]);
         
         $name          = $ext['name'];
         $name          = $this->_replaceTotemText($name);
         $lastVer       = $ext['version'];
         
         $moduleName    = $name;
         
         if (version_compare($currentVer, $lastVer, '<')):
            $status = '<img src="' . $this->getViewFileUrl('Totem_Core::images/update.gif') . '" alt="' . __("Update available") . '" title="'. __("Update available").'" />';
         endif;
      endif;
      
      // in case if module output disabled
      if ($this->scopeConfig->getValue('advanced/modules_disable_output/' . $moduleCode)):
         $status ='<img src="' . $this->getViewFileUrl('Totem_Core::images/bad.gif') . '" alt="' . __("Output disabled") . '" title="'. __("Output disabled") . '" />';
      endif;
      
      $moduleName = $status . ' ' . $moduleName;
      
      $field      = $fieldset->addField($moduleCode, 'label', [
         'name'  => 'dummy',
         'label' => $moduleName,
         'value' => $currentVer,
      ])->setRenderer($this->_getFieldRenderer());
      
      return $field->toHtml();
   }
   
   /**
    * @param $moduleName
    *
    * @return mixed
    */
   protected function _replaceTotemText($moduleName)
   {
      $moduleName = str_replace('Totem ', '', $moduleName);
      
      return $moduleName;
   }
}
