<?php
/**
 * @author     Bart de Groot
 * @company    Abovo Media
 * @copyright  Copyright (c) 2017 Abovo Media (http://www.abovomedia.nl)
 * @package    Totem_Cms
 */

namespace Totem\Cms\Helper;

use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Cms\Model\Wysiwyg\Images\Storage;
use Magento\Cms\Model\Page;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Catalog\Model\ImageUploader;

class Data extends AbstractHelper
{
   const IMAGE_FILETYPES = ['jpg', 'jpeg', 'png', 'gif'];
   
   /**
    * @var Storage
    */
   protected $storage;
   
   /**
    * @var Page
    */
   protected $page;
   
   /**
    * @var StoreManagerInterface
    */
   protected $storeManager;
   
   /**
    * @var ImageUploader
    */
   protected $imageUploader;
   
   /**
    * @param Storage $storage
    * @param Page $page
    * @param StoreManagerInterface $storeManager
    * @param ImageUploader $imageUploader
    */
   public function __construct(
      Storage $storage,
      Page $page,
      StoreManagerInterface $storeManager,
      ImageUploader $imageUploader
   ) {
      $this->storage       = $storage;
      $this->page          = $page;
      $this->storeManager  = $storeManager;
      $this->imageUploader = $imageUploader;
   }
   
   /**
    * Check if a cms block is visible or not
    *
    * @param \Magento\Cms\Model\Block $block
    *
    * @return boolean
    */
   public function isBlockVisible(\Magento\Cms\Model\Block $block)
   {
      $currentDate      = new \DateTime();
      $visibleFromDate  = new \DateTime($block->getVisibleFromDate());
      $visibleToDate    = new \DateTime($block->getVisibleToDate());
      
      if((is_null($block->getVisibleFromDate()) && is_null($block->getVisibleToDate())) || (($currentDate >= $visibleFromDate) && ($currentDate <= $visibleToDate))):
         return true;
      endif;
      
      return false;
   }
   
   /**
    * Check if the file type is custom or not
    *
    * @param $pathinfo
    *
    * @return bool
    */
   public function isCustomFileType($pathinfo)
   {
      if(!key_exists('extension', $pathinfo) || !in_array($pathinfo['extension'], $this->_getCustomFileTypes())):
         return false;
      endif;
      
      return true;
   }
   
   /**
    * Get the page content title
    *
    * @return bool|string
    *
    * @throws \Magento\Framework\Exception\LocalizedException
    */
   public function getPageContentTitle()
   {
      if(!($this->page->getId())):
         return false;
      endif;
      
      if(!($contentTitle = $this->page->getContentHeading())):
         return false;
      endif;
      
      return $contentTitle;
   }
   
   /**
    * Get the page content image
    *
    * @return bool|string
    *
    * @throws \Magento\Framework\Exception\LocalizedException
    */
   public function getPageContentImage()
   {
      if(!($this->page->getId())):
         return false;
      endif;
      
      if(!($contentImage = $this->page->getContentImage())):
         return false;
      endif;
      
      return $this->_getImageUrl($contentImage);
   }
   
   /**
    * Get a list of custom file types
    *
    * @return array
    */
   private function _getCustomFileTypes()
   {
      $fileTypes = $this->storage->getAllowedExtensions('file');
      
      foreach(self::IMAGE_FILETYPES as $imageType):
         if (($key = array_search($imageType, $fileTypes)) !== false):
            unset($fileTypes[$key]);
         endif;
      endforeach;
      
      return $fileTypes;
   }
   
   /**
    * Get image url
    *
    * @param string $image
    *
    * @return bool|string
    *
    * @throws \Magento\Framework\Exception\LocalizedException
    */
   private function _getImageUrl($image)
   {
      $url = false;
      if ($image):
         if (is_string($image)):
            $url = $this->storeManager->getStore()->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_MEDIA) . $this->imageUploader->getBasePath(). DIRECTORY_SEPARATOR . $image;
         else:
            throw new \Magento\Framework\Exception\LocalizedException(
               __('Something went wrong while getting the image url.')
            );
         
            return false;
         endif;
      endif;
      
      return $url;
   }
}