<?php
/**
 * @author     Bart de Groot
 * @company    Abovo Media
 * @copyright  Copyright (c) 2018 Abovo Media (http://www.abovomedia.nl)
 * @package    Totem_Catalog
 */

namespace Totem\Catalog\Helper;

use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Store\Model\ScopeInterface;
use Magento\Catalog\Model\Product\Gallery\ReadHandler as GalleryReadHandler;
use Magento\CatalogInventory\Api\StockRegistryInterface;

class Data extends AbstractHelper
{
   /**
    * @var ScopeConfigInterface
    */
   protected $scopeConfig;
   
   /**
    * @var GalleryReadHandler
    */
   protected $galleryReadHandler;
   
   /**
    * @var StockRegistryInterface
    */
   protected $stockRegistry;
   
   /**
    * @param \Magento\Framework\App\Helper\Context $context
    * @param GalleryReadHandler $galleryReadHandler
    * @param StockRegistryInterface $stockRegistry
    */
   public function __construct(
      \Magento\Framework\App\Helper\Context $context,
      GalleryReadHandler $galleryReadHandler,
      StockRegistryInterface $stockRegistry
   ) {
      $this->scopeConfig         = $context->getScopeConfig();
      $this->galleryReadHandler  = $galleryReadHandler;
      $this->stockRegistry       = $stockRegistry;
      
      parent::__construct($context);
   }
   
   /**
    * Get config value
    *
    * @return string
    */
   public function getConfig($configPath, $isSetFlag = false)
   {
      if($isSetFlag):
         return $this->scopeConfig->isSetFlag($configPath, ScopeInterface::SCOPE_STORE);
      endif;
      
      return $this->scopeConfig->getValue($configPath, ScopeInterface::SCOPE_STORE);
   }
   
   /**
    * Get recently viewed products limit
    *
    * @return string
    */
   public function getRecentlyViewedLimit()
   {
      return $this->getConfig('totem_catalog/recently_viewed/limit');
   }
   
   /**
    * Get recently viewed products limit
    *
    * @return string
    */
   public function getRecentlyViewedButtons()
   {
      $buttons = [];
      
      if($this->getConfig('totem_catalog/recently_viewed/show_button', true)):
         $buttons[] = 'view_product';
      endif;
      
      if($this->getConfig('totem_catalog/recently_viewed/show_wishlist', true)):
         $buttons[] = 'add_to_wishlist';
      endif;
      
      if($this->getConfig('totem_catalog/recently_viewed/show_compare', true)):
         $buttons[] = 'add_to_compare';
      endif;
      
      return implode(',', $buttons);
   }
   
   /**
    * Get hover image for product
    *
    * @return string
    */
   public function getHoverImage($item)
   {
      if(!$this->getConfig('totem_catalog/general/use_hover_images', true)):
         return false;
      endif;
      
      $this->_addGallery($item);
      if(!($images = $this->_getGalleryImages($item))):
         return false;
      endif;
      
      if($item->getDetailImage() != null && $item->getDetailImage() != 'no_selection'):
         return true;
      endif;
      
      foreach($images as $image):
         if($image->getFile() === $item->getImage()):
            continue;
         endif;
         
         $item->setData('detail_image', $image->getFile());
         
         return true;
      endforeach;
      
      return false;
   }
   
   /**
    * Get the product stock data and methods.
    *
    * @return StockRegistryInterface
    */
   public function getStockRegistry()
   {
      return $this->stockRegistry;
   }
   
   /**
    * Add the galery images to the product data
    *
    * @param $product
    */
   private function _addGallery($product)
   {
      $this->galleryReadHandler->execute($product);
   }
   
   /**
    * Get the gallery images
    *
    * @param \Magento\Catalog\Api\Data\ProductInterface $product
    *
    * @return mixed
    */
   private function _getGalleryImages(\Magento\Catalog\Api\Data\ProductInterface $product)
   {
      $images = $product->getMediaGalleryImages();
      
      if(!($images instanceof \Magento\Framework\Data\Collection)):
         return false;
      endif;
      
      if(count($images) < 2):
         return false;
      endif;
      
      return $images;
   }
}