<?php
/**
 * @author     Bart de Groot
 * @company    Abovo Media
 * @copyright  Copyright (c) 2017 Abovo Media (http://www.abovomedia.nl)
 * @package    Totem_Banner
 */

namespace Totem\Banner\Model\ResourceModel;

use Totem\Banner\Api\Data\BannerInterface;
use Magento\Framework\DB\Select;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\EntityManager\MetadataPool;
use Magento\Framework\EntityManager\EntityManager;
use Magento\Framework\Model\ResourceModel\Db\AbstractDb;
use Magento\Framework\Model\ResourceModel\Db\Context;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;

/**
 * Banner model
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class Banner extends AbstractDb
{
   /**
    * Store manager
    *
    * @var StoreManagerInterface
    */
   protected $_storeManager;
   
   /**
    * @var EntityManager
    */
   protected $entityManager;
   
   /**
    * @var MetadataPool
    */
   protected $metadataPool;
   
   /**
    * @param Context $context
    * @param StoreManagerInterface $storeManager
    * @param EntityManager $entityManager
    * @param MetadataPool $metadataPool
    * @param string $connectionName
    */
   public function __construct(
      Context $context,
      StoreManagerInterface $storeManager,
      EntityManager $entityManager,
      MetadataPool $metadataPool,
      $connectionName = null
   ) {
      $this->_storeManager = $storeManager;
      $this->entityManager = $entityManager;
      $this->metadataPool = $metadataPool;
      parent::__construct($context, $connectionName);
   }
   
   /**
    * Initialize resource model
    *
    * @return void
    */
   protected function _construct()
   {
      $this->_init('totem_banner', 'banner_id');
   }
   
   /**
    * @inheritDoc
    */
   public function getConnection()
   {
      return $this->metadataPool->getMetadata(BannerInterface::class)->getEntityConnection();
   }
   
   /**
    * Perform operations before object save
    *
    * @param AbstractModel $object
    * @return $this
    * @throws LocalizedException
    */
   protected function _beforeSave(AbstractModel $object)
   {
      return $this;
   }
   
   /**
    * Retrieve select object for load object data
    *
    * @param string $field
    * @param mixed $value
    * @param \Totem\Banner\Model\Banner|AbstractModel $object
    * @return Select
    */
   protected function _getLoadSelect($field, $value, $object)
   {
      $entityMetadata = $this->metadataPool->getMetadata(BannerInterface::class);
      $linkField = $entityMetadata->getLinkField();
      
      $select = parent::_getLoadSelect($field, $value, $object);
      
      if ($object->getStoreId()):
         $stores = [(int)$object->getStoreId(), Store::DEFAULT_STORE_ID];
         
         $select->join(
            ['tbs' => $this->getTable('totem_banner_store')],
            $this->getMainTable() . '.' . $linkField . ' = tbs.' . $linkField,
            ['store_id']
         )
                ->where('is_active = ?', 1)
                ->where('tbs.store_id in (?)', $stores)
                ->order('store_id DESC')
                ->limit(1);
      endif;
      
      return $select;
   }
   
   /**
    * Get store ids to which specified item is assigned
    *
    * @param int $id
    * @return array
    */
   public function lookupStoreIds($id)
   {
      $connection = $this->getConnection();
      
      $entityMetadata = $this->metadataPool->getMetadata(BannerInterface::class);
      $linkField = $entityMetadata->getLinkField();
      
      $select = $connection->select()
                           ->from(['tbs' => $this->getTable('totem_banner_store')], 'store_id')
                           ->join(
                              ['tb' => $this->getMainTable()],
                              'tbs.' . $linkField . ' = tb.' . $linkField,
                              []
                           )
                           ->where('tb.' . $entityMetadata->getIdentifierField()  . ' = :banner_id');
      
      return $connection->fetchCol($select, ['banner_id' => (int)$id]);
   }
   
   /**
    * @param AbstractModel $object
    * @return $this
    * @throws \Exception
    */
   public function save(AbstractModel $object)
   {
      $this->entityManager->save($object);
      
      return $this;
   }
   
   /**
    * @inheritDoc
    */
   public function delete(AbstractModel $object)
   {
      $this->entityManager->delete($object);
      
      return $this;
   }
}
