<?php
/**
 * @author     Bart de Groot
 * @company    Abovo Media
 * @copyright  Copyright (c) 2017 Abovo Media (http://www.abovomedia.nl)
 * @package    Totem_Banner
 */

namespace Totem\Banner\Model\ResourceModel;

use Magento\Store\Model\Store;

/**
 * Abstract collection of banners
 */
abstract class AbstractCollection extends \Magento\Framework\Model\ResourceModel\Db\Collection\AbstractCollection
{
   /**
    * Store manager
    *
    * @var \Magento\Store\Model\StoreManagerInterface
    */
   protected $storeManager;
   
   /**
    * @var \Magento\Framework\EntityManager\MetadataPool
    */
   protected $metadataPool;
   
   /**
    * @param \Magento\Framework\Data\Collection\EntityFactoryInterface $entityFactory
    * @param \Psr\Log\LoggerInterface $logger
    * @param \Magento\Framework\Data\Collection\Db\FetchStrategyInterface $fetchStrategy
    * @param \Magento\Framework\Event\ManagerInterface $eventManager
    * @param \Magento\Store\Model\StoreManagerInterface $storeManager
    * @param \Magento\Framework\EntityManager\MetadataPool $metadataPool
    * @param \Magento\Framework\DB\Adapter\AdapterInterface|null $connection
    * @param \Magento\Framework\Model\ResourceModel\Db\AbstractDb|null $resource
    */
   public function __construct(
      \Magento\Framework\Data\Collection\EntityFactoryInterface $entityFactory,
      \Psr\Log\LoggerInterface $logger,
      \Magento\Framework\Data\Collection\Db\FetchStrategyInterface $fetchStrategy,
      \Magento\Framework\Event\ManagerInterface $eventManager,
      \Magento\Store\Model\StoreManagerInterface $storeManager,
      \Magento\Framework\EntityManager\MetadataPool $metadataPool,
      \Magento\Framework\DB\Adapter\AdapterInterface $connection = null,
      \Magento\Framework\Model\ResourceModel\Db\AbstractDb $resource = null
   ) {
      $this->storeManager = $storeManager;
      $this->metadataPool = $metadataPool;
      parent::__construct($entityFactory, $logger, $fetchStrategy, $eventManager, $connection, $resource);
   }
   
   /**
    * Perform operations after collection load
    *
    * @param string $tableName
    * @param string|null $linkField
    * @return void
    */
   protected function performAfterLoad($tableName, $linkField)
   {
      $linkedIds = $this->getColumnValues($linkField);
      if (count($linkedIds)):
         $connection = $this->getConnection();
         $select = $connection->select()->from(['cms_entity_store' => $this->getTable($tableName)])
                              ->where('cms_entity_store.' . $linkField . ' IN (?)', $linkedIds);
         $result = $connection->fetchAll($select);
         if ($result):
            $storesData = [];
            foreach ($result as $storeData):
               $storesData[$storeData[$linkField]][] = $storeData['store_id'];
            endforeach;
            
            foreach ($this as $item):
               $linkedId = $item->getData($linkField);
               if (!isset($storesData[$linkedId])):
                  continue;
               endif;
               
               $storeIdKey = array_search(Store::DEFAULT_STORE_ID, $storesData[$linkedId], true);
               if ($storeIdKey !== false):
                  $stores = $this->storeManager->getStores(false, true);
                  $storeId = current($stores)->getId();
                  $storeCode = key($stores);
               else:
                  $storeId = current($storesData[$linkedId]);
                  $storeCode = $this->storeManager->getStore($storeId)->getCode();
               endif;
               $item->setData('_first_store_id', $storeId);
               $item->setData('store_code', $storeCode);
               $item->setData('store_id', $storesData[$linkedId]);
            endforeach;
         endif;
      endif;
   }
   
   /**
    * Add field filter to collection
    *
    * @param array|string $field
    * @param string|int|array|null $condition
    * @return $this
    */
   public function addFieldToFilter($field, $condition = null)
   {
      if ($field === 'store_id'):
         return $this->addStoreFilter($condition, false);
      endif;
      
      return parent::addFieldToFilter($field, $condition);
   }
   
   /**
    * Add filter by store
    *
    * @param int|array|Store $store
    * @param bool $withAdmin
    * @return $this
    */
   abstract public function addStoreFilter($store, $withAdmin = true);
   
   /**
    * Perform adding filter by store
    *
    * @param int|array|Store $store
    * @param bool $withAdmin
    * @return void
    */
   protected function performAddStoreFilter($store, $withAdmin = true)
   {
      if ($store instanceof Store):
         $store = [$store->getId()];
      endif;
      
      if (!is_array($store)):
         $store = [$store];
      endif;
      
      if ($withAdmin):
         $store[] = Store::DEFAULT_STORE_ID;
      endif;
      
      $this->addFilter('store', ['in' => $store], 'public');
   }
   
   /**
    * Join store relation table if there is store filter
    *
    * @param string $tableName
    * @param string|null $linkField
    * @return void
    */
   protected function joinStoreRelationTable($tableName, $linkField)
   {
      if ($this->getFilter('store')):
         $this->getSelect()->join(
            ['store_table' => $this->getTable($tableName)],
            'main_table.' . $linkField . ' = store_table.' . $linkField,
            []
         )->group(
            'main_table.' . $linkField
         );
      endif;
      parent::_renderFiltersBefore();
   }
   
   /**
    * Get SQL for get banner count
    *
    * Extra GROUP BY strip added.
    *
    * @return \Magento\Framework\DB\Select
    */
   public function getSelectCountSql()
   {
      $countSelect = parent::getSelectCountSql();
      $countSelect->reset(\Magento\Framework\DB\Select::GROUP);
      
      return $countSelect;
   }
}
