<?php
/**
 * @author     Bart de Groot
 * @company    Abovo Media
 * @copyright  Copyright (c) 2017 Abovo Media (http://www.abovomedia.nl)
 * @package    Totem_Banner
 */

namespace Totem\Banner\Model;

use Totem\Banner\Api\Data\BannerInterface;
use Magento\Framework\DataObject\IdentityInterface;
use Magento\Framework\Model\AbstractModel;

/**
 * Banner model
 *
 * @method Banner setStoreId(array $storeId)
 * @method array getStoreId()
 */
class Banner extends AbstractModel implements BannerInterface, IdentityInterface
{
   /**
    * Banner cache tag
    */
   const CACHE_TAG = 'totem_b';
   
   /**#@+
    * Banner's statuses
    */
   const STATUS_ENABLED = 1;
   const STATUS_DISABLED = 0;
   
   /**#@-*/
   
   /**#@-*/
   protected $_cacheTag = self::CACHE_TAG;
   
   /**
    * Prefix of model events names
    *
    * @var string
    */
   protected $_eventPrefix = 'totem_banner';
   
   /**
    * @var \Totem\Banner\Model\ImageUploader
    */
   protected $imageUploader;
   
   /**
    * @param \Magento\Framework\Model\Context $context
    * @param \Magento\Framework\Registry $registry
    * @param \Magento\Framework\Model\ResourceModel\AbstractResource $resource
    * @param \Magento\Framework\Data\Collection\AbstractDb $resourceCollection
    * @param \Totem\Banner\Model\ImageUploader $imageUploader
    * @param array $data
    */
   public function __construct(
      \Magento\Framework\Model\Context $context,
      \Magento\Framework\Registry $registry,
      \Totem\Banner\Model\ImageUploader $imageUploader,
      \Magento\Framework\Model\ResourceModel\AbstractResource $resource = null,
      \Magento\Framework\Data\Collection\AbstractDb $resourceCollection = null,
      array $data = []
   ) {
      $this->imageUploader = $imageUploader;
      
      parent::__construct($context, $registry, $resource, $resourceCollection, $data);
   }
   
   /**
    * @return void
    */
   protected function _construct()
   {
      $this->_init(\Totem\Banner\Model\ResourceModel\Banner::class);
   }
   
   /**
    * Get identities
    *
    * @return array
    */
   public function getIdentities()
   {
      return [self::CACHE_TAG . '_' . $this->getId()];
   }
   
   /**
    * Retrieve id
    *
    * @return int
    */
   public function getId()
   {
      return $this->getData(self::BANNER_ID);
   }
   
   /**
    * Retrieve title
    *
    * @return string
    */
   public function getTitle()
   {
      return $this->getData(self::TITLE);
   }
   
   /**
    * Retrieve subtitle
    *
    * @return string
    */
   public function getSubtitle()
   {
      return $this->getData(self::SUBTITLE);
   }
   
   /**
    * Retrieve link type
    *
    * @return string
    */
   public function getLinkType()
   {
      return $this->getData(self::LINK_TYPE);
   }
   
   /**
    * Retrieve link category
    *
    * @return string
    */
   public function getLinkCategory()
   {
      return $this->getData(self::LINK_CATEGORY);
   }
   
   /**
    * Retrieve link page
    *
    * @return string
    */
   public function getLinkPage()
   {
      return $this->getData(self::LINK_PAGE);
   }
   
   /**
    * Retrieve link custom
    *
    * @return string
    */
   public function getLinkCustom()
   {
      return $this->getData(self::LINK_CUSTOM);
   }
   
   /**
    * Retrieve link new window
    *
    * @return boolean
    */
   public function getLinkNewWindow()
   {
      return $this->getData(self::LINK_NEW_WINDOW);
   }
   
   /**
    * Retrieve show button
    *
    * @return boolean
    */
   public function getShowButton()
   {
      return $this->getData(self::SHOW_BUTTON);
   }
   
   /**
    * Retrieve button label
    *
    * @return string
    */
   public function getButtonLabel()
   {
      return $this->getData(self::BUTTON_LABEL);
   }
   
   /**
    * Retrieve type
    *
    * @return string
    */
   public function getType()
   {
      return $this->getData(self::TYPE);
   }
   
   /**
    * Retrieve image
    *
    * @return string
    */
   public function getImage()
   {
      return $this->getData(self::IMAGE);
   }
   
   /**
    * Retrieve vide
    *
    * @return string
    */
   public function getVideo()
   {
      return $this->getData(self::VIDEO);
   }
   
   /**
    * Retrieve video json
    *
    * @return string
    */
   public function getVideoJson()
   {
      return $this->getData(self::VIDEO_JSON);
   }
   
   /**
    * Retrieve content
    *
    * @return string
    */
   public function getContent()
   {
      return $this->getData(self::CONTENT);
   }
   
   /**
    * Get image url
    *
    * @return bool|string
    *
    * @throws \Magento\Framework\Exception\LocalizedException
    */
   public function getImageUrl()
   {
      $url = false;
      $image = $this->getData(self::IMAGE);
      if ($image):
         if (is_string($image)):
            $url = $this->imageUploader->getBaseUrl() . $this->imageUploader->getBasePath(). DIRECTORY_SEPARATOR . $image;
         else:
            throw new \Magento\Framework\Exception\LocalizedException(
               __('Something went wrong while getting the image url.')
            );
         endif;
      endif;
      
      return $url;
   }
   
   /**
    * Retrieve creation time
    *
    * @return string
    */
   public function getCreationTime()
   {
      return $this->getData(self::CREATION_TIME);
   }
   
   /**
    * Retrieve update time
    *
    * @return string
    */
   public function getUpdateTime()
   {
      return $this->getData(self::UPDATE_TIME);
   }
   
   /**
    * Is active
    *
    * @return bool
    */
   public function isActive()
   {
      return (bool)$this->getData(self::IS_ACTIVE);
   }
   
   /**
    * Set ID
    *
    * @param int $id
    * @return BannerInterface
    */
   public function setId($id)
   {
      return $this->setData(self::BANNER_ID, $id);
   }
   
   /**
    * Set title
    *
    * @param string $title
    * @return BannerInterface
    */
   public function setTitle($title)
   {
      return $this->setData(self::TITLE, $title);
   }
   
   /**
    * Set subtitle
    *
    * @param string $subtitle
    * @return BannerInterface
    */
   public function setSubtitle($subtitle)
   {
      return $this->setData(self::SUBTITLE, $subtitle);
   }
   
   /**
    * Set link type
    *
    * @param string $linkType
    * @return BannerInterface
    */
   public function setLinkType($linkType)
   {
      return $this->setData(self::LINK_TYPE, $linkType);
   }
   
   /**
    * Set link category
    *
    * @param string $linkCategory
    * @return BannerInterface
    */
   public function setLinkCategory($linkCategory)
   {
      return $this->setData(self::LINK_CATEGORY, $linkCategory);
   }
   
   /**
    * Set link page
    *
    * @param string $linkPage
    * @return BannerInterface
    */
   public function setLinkPage($linkPage)
   {
      return $this->setData(self::LINK_PAGE, $linkPage);
   }
   
   /**
    * Set link custom
    *
    * @param string $linkCustom
    * @return BannerInterface
    */
   public function setLinkCustom($linkCustom)
   {
      return $this->setData(self::LINK_CUSTOM, $linkCustom);
   }
   
   /**
    * Set link new window
    *
    * @param string $linkNewWindow
    * @return BannerInterface
    */
   public function setLinkNewWindow($linkNewWindow)
   {
      return $this->setData(self::LINK_NEW_WINDOW, $linkNewWindow);
   }
   
   /**
    * Set show button
    *
    * @param string $showButton
    * @return BannerInterface
    */
   public function setShowButton($showButton)
   {
      return $this->setData(self::SHOW_BUTTON, $showButton);
   }
   
   /**
    * Set button label
    *
    * @param string $buttonLabel
    * @return BannerInterface
    */
   public function setButtonLabel($buttonLabel)
   {
      return $this->setData(self::BUTTON_LABEL, $buttonLabel);
   }
   
   /**
    * Set type
    *
    * @param string $type
    * @return BannerInterface
    */
   public function setType($type)
   {
      return $this->setData(self::TYPE, $type);
   }
   
   /**
    * Set image
    *
    * @param string $image
    * @return BannerInterface
    */
   public function setImage($image)
   {
      return $this->setData(self::IMAGE, $image);
   }
   
   /**
    * Set video
    *
    * @param string $video
    * @return BannerInterface
    */
   public function setVideo($video)
   {
      return $this->setData(self::VIDEO, $video);
   }
   
   /**
    * Set video json
    *
    * @param string $videoJson
    * @return BannerInterface
    */
   public function setVideoJson($videoJson)
   {
      return $this->setData(self::VIDEO_JSON, $videoJson);
   }
   
   /**
    * Set content
    *
    * @param string $content
    * @return BannerInterface
    */
   public function setContent($content)
   {
      return $this->setData(self::CONTENT, $content);
   }
   
   /**
    * Set creation time
    *
    * @param string $creationTime
    * @return BannerInterface
    */
   public function setCreationTime($creationTime)
   {
      return $this->setData(self::CREATION_TIME, $creationTime);
   }
   
   /**
    * Set update time
    *
    * @param string $updateTime
    * @return BannerInterface
    */
   public function setUpdateTime($updateTime)
   {
      return $this->setData(self::UPDATE_TIME, $updateTime);
   }
   
   /**
    * Set is active
    *
    * @param bool|int $isActive
    * @return BannerInterface
    */
   public function setIsActive($isActive)
   {
      return $this->setData(self::IS_ACTIVE, $isActive);
   }
   
   /**
    * Receive page store ids
    *
    * @return int[]
    */
   public function getStores()
   {
      return $this->hasData('stores') ? $this->getData('stores') : $this->getData('store_id');
   }
   
   /**
    * Prepare banner's statuses.
    *
    * @return array
    */
   public function getAvailableStatuses()
   {
      return [self::STATUS_ENABLED => __('Enabled'), self::STATUS_DISABLED => __('Disabled')];
   }
}
