<?php
/**
 * @author     Bart de Groot
 * @company    Abovo Media
 * @copyright  Copyright (c) 2017 Abovo Media (http://www.abovomedia.nl)
 * @package    Totem_Banner
 */

namespace Totem\Banner\Block\Widget;

use Magento\Framework\App\ObjectManager;
use Magento\Framework\Serialize\Serializer\Json;

/**
 * Banner Widget
 */
class Banner extends \Magento\Framework\View\Element\Template implements \Magento\Widget\Block\BlockInterface
{
   protected $_template = 'widget/banner.phtml';
   
   /**
    * Storage for used widgets
    *
    * @var array
    */
   protected static $_widgetUsageMap = [];
   
   /**
    * Banner Factory
    *
    * @var \Totem\Banner\Model\BannerFactory
    */
   protected $bannerFactory;
   
   /**
    * Category Repository Interface
    *
    * @var \Magento\Catalog\Api\CategoryRepositoryInterface
    */
   protected $categoryRepository;
   
   /**
    * CMS Page Helper
    *
    * @var Page
    */
   protected $cmsPageHelper;
   
   /**
    * @var Json
    */
   protected $serializer;
   
   /**
    * @param \Magento\Framework\View\Element\Template\Context $context
    * @param \Totem\Banner\Model\BannerFactory $bannerFactory
    * @param \Magento\Cms\Helper\Page $cmsPageHelper
    * @param \Magento\Catalog\Api\CategoryRepositoryInterface $categoryRepository
    * @param Json $serializer
    * @param array $data
    */
   public function __construct(
      \Magento\Framework\View\Element\Template\Context $context,
      \Totem\Banner\Model\BannerFactory $bannerFactory,
      \Magento\Cms\Helper\Page $cmsPageHelper,
      \Magento\Catalog\Api\CategoryRepositoryInterface $categoryRepository,
      Json $serializer = null,
      array $data = []
   ) {
      parent::__construct($context, $data);
      
      $this->bannerFactory       = $bannerFactory;
      $this->cmsPageHelper       = $cmsPageHelper;
      $this->categoryRepository  = $categoryRepository;
      $this->serializer          = $serializer ?: ObjectManager::getInstance()->get(Json::class);
   }
   
   /**
    * Check if the banner is available
    *
    * @return bool
    * @throws \Magento\Framework\Exception\NoSuchEntityException
    */
   public function isBannerAvailable()
   {
      $bannerId = $this->getData('banner_id');
      if(!$bannerId):
         return false;
      endif;
      
      $storeId = $this->_storeManager->getStore()->getId();
      /** @var \Totem\Banner\Model\Banner $banner */
      $banner = $this->bannerFactory->create();
      $banner->setStoreId($storeId)->load($bannerId);
      if(!$banner->isActive()):
         return false;
      endif;
      
      return true;
   }
   
   /**
    * Prevent banners recursion if needed
    *
    * @return $this
    */
   protected function _beforeToHtml()
   {
      parent::_beforeToHtml();
      $bannerId = $this->getData('banner_id');
      $bannerHash = get_class($this) . $bannerId;
      
      if (isset(self::$_widgetUsageMap[$bannerHash])):
         return $this;
      endif;
      self::$_widgetUsageMap[$bannerHash] = true;
      
      if ($bannerId):
         $storeId = $this->_storeManager->getStore()->getId();
         /** @var \Totem\Banner\Model\Banner $banner */
         $banner = $this->bannerFactory->create();
         $banner->setStoreId($storeId)->load($bannerId);
         if ($banner->isActive()):
            $this->setTitle($banner->getTitle() ?? '');
            $this->setSubtitle($banner->getSubtitle() ?? '');
            $this->setLink($this->_getLink($banner));
            $this->setLinkNewWindow($banner->getLinkNewWindow());
            $this->setImageUrl($banner->getImageUrl());
            $this->setContent($banner->getContent());
            $this->setShowButton($banner->getShowButton());
            if($banner->getShowButton()):
               $this->setButtonLabel($banner->getButtonLabel());
            endif;
            $this->setType($banner->getType());
            $this->setVideo($banner->getVideo());
            $this->setVideoJson($this->_getVideoJson($banner->getVideoJson()));
            $this->setEmbedHtml($this->_getEmbedHtml($banner->getVideo()));
         endif;
      endif;
      
      unset(self::$_widgetUsageMap[$bannerHash]);
      
      return $this;
   }
   
   /**
    * Generate the banner link based on the link type
    *
    * @return string $url
    */
   private function _getLink($banner)
   {
      switch ($banner->getLinkType()):
         case 'no_link':
         default:
            $url = false;
      
            break;
         case \Totem\Banner\Api\Data\BannerInterface::LINK_CUSTOM:
            $url = ($banner->getLinkCustom() ?? '');
            if (strpos($url, '://') === false):
               $url = $this->_urlBuilder->getDirectUrl($url != '/' ? $url : '');
            endif;
         
            break;
         case \Totem\Banner\Api\Data\BannerInterface::LINK_PAGE:
            if($pageIdentifier = $banner->getLinkPage()):
               $url = $this->cmsPageHelper->getPageUrl($pageIdentifier);
            endif;
         
            break;
         case \Totem\Banner\Api\Data\BannerInterface::LINK_CATEGORY:
            if($categoryId = $banner->getLinkCategory()):
               if($this->_getCategory($categoryId)):
                  $url = $this->_getCategory($categoryId)->getUrl();
               endif;
            endif;
         
            break;
      endswitch;
      
      if(!isset($url) || is_null($url)):
         $url = $this->_storeManager->getStore()->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_WEB);
      endif;
   
      return $url;
   }
   
   /**
    * Get the category by id
    *
    * @return \Magento\Catalog\Api\Data\CategoryInterface $category
    */
   private function _getCategory($categoryId)
   {
      try
      {
         $category = $this->categoryRepository->get($categoryId, $this->_storeManager->getStore()->getId());
      }
      catch(\Magento\Framework\Exception\NoSuchEntityException $e)
      {
         return false;
      }
      
      return $category;
   }
   
   /**
    * Retrieve the video json
    *
    * @return array
    */
    private function _getVideoJson($json)
   {
      if(!$json || empty($json)):
         return false;
      endif;
      
      $json = htmlspecialchars_decode($json, ENT_QUOTES);
      
      return $this->serializer->unserialize($json);
   }
   
   /**
    * Retrieve the video embed html
    *
    * @return string
    */
   private function _getEmbedHtml($url)
   {
      if (strpos($url, 'youtu') !== false) :
   
         //extract the ID
         parse_str(parse_url($url, PHP_URL_QUERY), $videoId);
   
         //the ID of the YouTube URL
         if(isset($videoId['v'])):
            $id = $videoId['v'];
            //return the embed code
            return '<iframe src="https://youtube.com/embed/' . $id . '?autoplay=1&playlist=' . $id . '&controls=0&showinfo=0&autohide=1&disablekb=1&mute=1&loop=1&modestbranding=1&showinfo=0" frameborder="0" height="100%" width="100%" ></iframe>';
         endif;
      elseif (strpos($url, 'vimeo') !== false) :
            
         //extract the ID
         preg_match("/(https?:\/\/)?(www\.)?(player\.)?vimeo\.com\/([a-z]*\/)*([0-9]{6,11})[?]?.*/", $url, $videoId);

         //the ID of the Vimeo URL
         if(isset($videoId[5])):
            $id = $videoId[5];
            //return the embed code
            return '<iframe src="https://player.vimeo.com/video/' . $id . '?background=1&autoplay=1&loop=1&byline=0&title=0" frameborder="0" height="100%" width="100%" ></iframe>';
         endif;
      endif;
      
      return false;
   }
}
