<?php
/**
 * @author     Bart de Groot
 * @company    Abovo Media
 * @copyright  Copyright (c) 2017 Abovo Media (http://www.abovomedia.nl)
 * @package    Totem_Banner
 */

namespace Totem\Banner\Block\Adminhtml\Banner\Widget;

/**
 * Banner chooser for banner widget
 */
class Chooser extends \Magento\Backend\Block\Widget\Grid\Extended
{
   /**
    * @var \Totem\Banner\Model\Banner
    */
   protected $banner;
   
   /**
    * @var \Totem\Banner\Model\BannerFactory
    */
   protected $bannerFactory;
   
   /**
    * @var \Totem\Banner\Model\ResourceModel\Banner\CollectionFactory
    */
   protected $collectionFactory;
   
   /**
    * @var \Totem\Banner\Model\Banner\Source\LinkType
    */
   protected $linkType;
   
   /**
    * @param \Magento\Backend\Block\Template\Context $context
    * @param \Magento\Backend\Helper\Data $backendHelper
    * @param \Totem\Banner\Model\Banner $banner
    * @param \Totem\Banner\Model\BannerFactory $bannerFactory
    * @param \Totem\Banner\Model\ResourceModel\Banner\CollectionFactory $collectionFactory
    * @param \Totem\Banner\Model\Banner\Source\LinkType $linkType
    * @param array $data
    */
   public function __construct(
      \Magento\Backend\Block\Template\Context $context,
      \Magento\Backend\Helper\Data $backendHelper,
      \Totem\Banner\Model\Banner $banner,
      \Totem\Banner\Model\BannerFactory $bannerFactory,
      \Totem\Banner\Model\ResourceModel\Banner\CollectionFactory $collectionFactory,
      \Totem\Banner\Model\Banner\Source\LinkType $linkType,
      array $data = []
   ) {
      $this->linkType            = $linkType;
      $this->banner              = $banner;
      $this->bannerFactory       = $bannerFactory;
      $this->collectionFactory   = $collectionFactory;
      parent::__construct($context, $backendHelper, $data);
   }
   
   /**
    * Block construction, prepare grid params
    *
    * @return void
    */
   protected function _construct()
   {
      parent::_construct();
      
      $this->setDefaultSort('banner_id');
      $this->setDefaultDir('ASC');
      $this->setUseAjax(true);
      $this->setDefaultFilter(['chooser_is_active' => '1']);
   }
   
   /**
    * Prepare chooser element HTML
    *
    * @param \Magento\Framework\Data\Form\Element\AbstractElement $element Form Element
    * @return \Magento\Framework\Data\Form\Element\AbstractElement
    */
   public function prepareElementHtml(\Magento\Framework\Data\Form\Element\AbstractElement $element)
   {
      $uniqId     = $this->mathRandom->getUniqueHash($element->getId());
      $sourceUrl  = $this->getUrl('totem_banner/banner_widget/chooser', ['uniq_id' => $uniqId]);
      
      $chooser = $this->getLayout()->createBlock(
         \Magento\Widget\Block\Adminhtml\Widget\Chooser::class
      )->setElement(
         $element
      )->setConfig(
         $this->getConfig()
      )->setFieldsetId(
         $this->getFieldsetId()
      )->setSourceUrl(
         $sourceUrl
      )->setUniqId(
         $uniqId
      );
      
      if ($element->getValue()):
         $banner = $this->bannerFactory->create()->load((int)$element->getValue());
         if ($banner->getId()):
            $chooser->setLabel($this->escapeHtml($banner->getIdentifier()));
         endif;
      endif;
      
      $element->setData('after_element_html', $chooser->toHtml());
      
      return $element;
   }
   
   /**
    * Grid Row JS Callback
    *
    * @return string
    */
   public function getRowClickCallback()
   {
      $chooserJsObject = $this->getId();
      $js = '
            function (grid, event) {
                var trElement = Event.findElement(event, "tr");
                var bannerId = trElement.down("td").innerHTML.replace(/^\s+|\s+$/g,"");
                var bannerTitle = trElement.down("td").next().innerHTML;
                ' .
            $chooserJsObject .
            '.setElementValue(bannerId);
                ' .
            $chooserJsObject .
            '.setElementLabel(bannerTitle);
                ' .
            $chooserJsObject .
            '.close();
            }
        ';
      
      return $js;
   }
   
   /**
    * Prepare banners collection
    *
    * @return \Magento\Backend\Block\Widget\Grid\Extended
    */
   protected function _prepareCollection()
   {
      $this->setCollection($this->collectionFactory->create());
      
      return parent::_prepareCollection();
   }
   
   /**
    * Prepare columns for pages grid
    *
    * @return $this
    */
   protected function _prepareColumns()
   {
      $this->addColumn(
         'chooser_id',
         [
            'header' => __('ID'),
            'align' => 'right',
            'index' => 'banner_id',
            'width' => 50,
            'header_css_class' => 'col-id',
            'column_css_class' => 'col-id'
         ]
      );
      
      $this->addColumn(
         'chooser_identifier',
         [
            'header' => __('Identifier'),
            'index' => 'identifier',
            'header_css_class' => 'col-identifier',
            'column_css_class' => 'col-identifier'
         ]
      );
      
      $this->addColumn(
         'chooser_title',
         [
            'header' => __('Title'),
            'index' => 'title',
            'header_css_class' => 'col-title',
            'column_css_class' => 'col-title'
         ]
      );
      
      $this->addColumn(
         'chooser_subtitle',
         [
            'header' => __('Subtitle'),
            'index' => 'subtitle',
            'header_css_class' => 'col-url',
            'column_css_class' => 'col-url'
         ]
      );
      
      $this->addColumn(
          'chooser_link_type',
          [
              'header' => __('Link Type'),
              'index' => 'link_type',
              'type' => 'options',
              'options' => $this->linkType->getOptions(),
              'header_css_class' => 'col-link-type',
              'column_css_class' => 'col-link-type'
          ]
      );
      
      $this->addColumn(
         'chooser_is_active',
         [
            'header' => __('Status'),
            'index' => 'is_active',
            'type' => 'options',
            'options' => $this->banner->getAvailableStatuses(),
            'header_css_class' => 'col-status',
            'column_css_class' => 'col-status'
         ]
      );
      
      return parent::_prepareColumns();
   }
   
   /**
    * Get grid url
    *
    * @return string
    */
   public function getGridUrl()
   {
      return $this->getUrl('totem_banner/banner_widget/chooser', ['_current' => true]);
   }
}
